option language:c
;This program goes directly to PMODE from RMODE.
;This program will not work if a VCPI or DPMI driver is loaded.
;So you can not have EMM386,QEMM,386MAX,Windoze, OS/2 or anything loaded
;while using this little example (don't worry, eventually this will
;run under those programs but not yet).

; Stage #3
; - all RMODE IRQs are redir to PMODE (only if needed)
; - added DPMI funcs 300h-302h,305h-306h

; Notes:
; - Now ALL IRQs will occur in PMODE if a handler for it is setup.

; New funcs see:
; irq0-irqf: and rm_irq_redir proc - passes IRQs to PMODE
; int31h proc - DPMI func request handle
; _300h => _302h - DPMI func call rmode INT/PROC
; _305h => _306h - DPMI func get raw switching routines

.386p   ;allow USE32 and 386 instructions
;-----------------------------------------------------------------------------
;some EQUates
;-----------------------------------------------------------------------------
stack16siz equ 1024*2
stack32siz equ 1024*4

bptr equ byte ptr
wptr equ word ptr
dptr equ dword ptr
fptr equ fword ptr
qptr equ qword ptr

ERROR equ -1

;-----------------------------------------------------------------------------
;define all segments that will be needed
;-----------------------------------------------------------------------------
code16seg segment use16 'code'
code16seg ends
data16seg segment use16
data16seg ends
stack16seg segment use16 stack
  db stack16siz dup (?)
  TOS16 equ $   ;top of stack
stack16seg ends

code32seg segment use32 'code'
code32seg ends
data32seg segment use32
data32seg ends
stack32seg segment use32 'stack'
  db stack32siz dup (?)
  TOS32 equ $   ;top of stack
stack32seg ends

;put all segments in one
segs16 group code16seg,data16seg,stack16seg
segs32 group code32seg,data32seg,stack32seg

assume cs:segs16,ds:segs16,ss:segs16,es:NOTHING

;-----------------------------------------------------------------------------
;data structs needed
;-----------------------------------------------------------------------------
desc struct   ;define our descriptor structure
  limit_lo dw ?    ;limit bits 15-0
  base_lo dw ?     ;base bits 15-0
  base_mid db ?    ;base bits 23-16
  type1 db ?       ;type of selector
  limit_hi db ?    ;limit bits 19-16 and other info
  base_hi db ?     ;base bits 31-24
desc ends

callstruct struct  ;used in calling rmode
  _edi dd ?     ;0
  _esi dd ?     ;4
  _ebp dd ?     ;8
  _res dd ?     ;0ch reserved
  _ebx dd ?     ;10h
  _edx dd ?     ;14h
  _ecx dd ?     ;18h
  _eax dd ?     ;1ch
  _flg dw ?     ;20h flags
  _es dw ?      ;22h segments (NOT selectors)
  _ds dw ?      ;24h "
  _fs dw ?      ;26h "
  _gs dw ?      ;28h "
  _ip dw ?      ;2ah ignored when calling INT
  _cs dw ?      ;2ch "
  _sp dw ?      ;2eh must be 0 to use system stacks
  _ss dw ?      ;30h "
callstruct ends

;-----------------------------------------------------------------------------
;16bit data segment
;-----------------------------------------------------------------------------
data16seg segment use16  ;start defining our 16bit data

;define our GDTR
  gdtr label fword
  gdt_limit dw gdt_size-1
  gdt_addr dd ?
;define our IDTR
  idtr label fword
  idt_limit dw idt_size-1
  idt_addr dd ?
;define the default IDTR (this is used while in RMODE)
  rm_idtr label fword
  dw 3ffh  ;limit
  dd 0     ;addr

;-----------------------------------------------------------------------------
;define our GDT
;-----------------------------------------------------------------------------
  gdt_start equ $
  null desc <0,0,0,0,0,0>
    ;this is the 1st GDT entry called NULL which we can not use
  code16 desc <0ffffh,?,?,10011010b,0h,?>
    ;Limit=64 KBs (this is a 16bit segment so only IP is used while running)
    ;Base=? (this will be setup later in the program)
    ;10011?10b = P=1 (present), DPL=0, S=1(code/data segment) T=1 (code)
    ;            C=1(conforming), R=1 (readable), A=0 (not accessed)
    ;0h = limit bits 19-16=0, G=0 (1byte granularity), D=0 (16bit segment)
  data16 desc <0ffffh,?,?,10010010b,0h,?>
    ;Limit=64 KBs
    ;Base=? (this will be setup later in the program)
    ;10010010b = P=1 (present), DPL=0, S=1(code/data segment) T=0 (data)
    ;            E=0 (do not expand down), W=1 (writable), A=0 (not accessed)
    ;0h = limit bits 19-16=0, G=0 (1byte granularity), D=0 (16bit segment)
  vid desc <0ffffh,8000h,0bh,10010010b,0h,0>
    ;Limit=64 KBs
    ;Base=0b8000h (makes it easy to access video RAM while in PMODE-16)
    ;10010010b = P=1 (present), DPL=0, S=1(code/data segment) T=0 (data)
    ;            E=0 (do not expand down), W=1 (writable), A=0 (not accessed)
    ;0h = limit bits 19-16=0, G=0 (1byte granularity), D=0 (16bit segment)
  code32 desc <0ffffh,?,?,10011010b,11001111b,?>
    ;Limit=4 GBs (0ffffh * 4k = 4 GBs)
    ;Base=?
    ;     the beginning of memory)
    ;10011110b = P=1 (present), DPL=0, S=1(code/data segment) T=1 (code)
    ;            C=1(conforming), R=1 (readable), A=0 (not accessed)
    ;11001111b = limit bits 19-16=0fh, G=1 (4k granularity), D=1 (32bit segment)
  data32 desc <0ffffh,?,?,10010010b,11001111b,?>
    ;Limit=4 GBs
    ;Base=?
    ;10010010b = P=1 (present), DPL=0, S=1(code/data segment) T=0 (data)
    ;            E=0 (do not expand down), W=1 (writable), A=0 (not accessed)
    ;11001111b = limit bits 19-16=0fh, G=1 (4k granularity), D=1 (32bit segment)
  zero32 desc <0ffffh,0,0,10010010b,11001111b,0>
    ;Limit=4 GBs
    ;Base=0h (easy access to rmode memory)
    ;10010010b = P=1 (present), DPL=0, S=1(code/data segment) T=0 (data)
    ;            E=0 (do not expand down), W=1 (writable), A=0 (not accessed)
    ;11001111b = limit bits 19-16=0fh, G=1 (4k granularity), D=1 (32bit segment)
  ldt desc <0,0,0,10000010b,0h,0>
    ;this is what will be loaded into our LDTR.  Because we don't want
    ;to use the LDT this is setup so that our LDT is empty
    ;Limit=1 byte (limits of 0 are not possible)
    ;Base=0
    ;10000010b = P=1 (present), DPL=0, S=0(system segment), TYPE=2 (LDT)
    ;       0h = limit bits 19-16=0, G=0 (1byte granularity), D=0 (16bit segment)
  gdt_size equ ($-gdt_start)

;-----------------------------------------------------------------------------
;define our IDT (pmode)
;-----------------------------------------------------------------------------
  idt_start equ $
  idt desc 256 dup (<?,selcode16,0,10001110b,0,0>)
  idt_size equ ($-idt_start)

;define our intrmatrix
ints db 256 dup (0cch)                 ;this is what each of the default
                                       ;INT descriptors pts to
                                       ;0cch = int 3 (one-byte opcode)
                                       ;so ALL INTs goto the INT 3 handler
                                       ;which will direct the INT to rmode

;define our selectors as EQU
  selcode16 equ (offset code16-gdt_start)
  seldata16 equ (offset data16-gdt_start)
  selvid equ    (offset vid -gdt_start)
  seldata32 equ (offset data32-gdt_start)
  selzero32 equ (offset zero32-gdt_start)
  selcode32 equ (offset code32-gdt_start)
  selldt equ    (offset ldt   -gdt_start)

;define our selectors as DW
  _selcode16  dw selcode16
  _seldata16  dw seldata16
  _selvid     dw selvid
  _seldata32  dw seldata32
  _selzero32  dw selzero32
  _selcode32  dw selcode32
  _selldt     dw selldt

;define our raw switching routines
  pm_2_rm dd offset pm_2_rm_rx         ;by default our switching routines
          dd selcode16                 ;will be for raw/xms but later it may
  rm_2_pm dw offset rm_2_pm_rx         ;become VCPI
          dw segs16
;define our save/restore state routines
  sr_state_pm dd offset sr_state_pm_rx
              dd selcode16
  sr_state_rm dw offset sr_state_rm_rx
              dw segs16

align 4
;-----------------------------------------------------------------------------
;define our pm 2 rm stacks
  pm2rm_stkend dw ?       ;Last avail stack   (ss)
  pm2rm_stkcur dw ?       ;Current            (ss)
  pm2rm_stklen dw 80h     ;incremental(para)  (sp)
  pm2rm_stks dw 8         ;# of stacks
;define our rm 2 pm stacks
  rm2pm_stkend dw ?       ;Last avail stack   (ss)
  rm2pm_stkcur dw ?       ;Current            (ss)
  rm2pm_stklen dw 40h     ;incremental(para)  (sp)
  rm2pm_stks dw 8         ;# of stacks
;-----------------------------------------------------------------------------

;define temp variables needed
  tmpd1 dd ?
;define other variables needed
irq_table dw offset irq0
          dw offset irq1
          dw offset irq2
          dw offset irq3
          dw offset irq4
          dw offset irq5
          dw offset irq6
          dw offset irq7
          dw offset irq8
          dw offset irq9
          dw offset irqa
          dw offset irqb
          dw offset irqc
          dw offset irqd
          dw offset irqe
          dw offset irqf

old_irqs dd 16 dup (?)   ;old rmode IRQ handlers
irqset dw 0              ;each bit is set if an IRQ handler is setup in PMODE
                         ;for that IRQ

old_sp dw ?              ;used by 30xh DPMI funcs

mode db 0                ;defines what drivers are installed
  MODE_RAW equ 0         ;none
  MODE_XMS equ 1         ;XMS (himem.sys)
  MODE_VCPI equ 2        ;VCPI (Emm386.exe)  Not supported yet
  MODE_DPMI equ 3        ;DPMI (Winblows)  Not supported yet

xms_max dw 1024  ;Max RAM to Alloc is 1024 KBs
xms_min dw 64    ;Min RAM to Alloc is 64 KBs
xms_left dw ?    ;int 15h will report this much is free
xms_base dd ?    ;the start of our XMS RAM we have alloc
xms_size dd ?    ;the size of our XMS RAM
xms_call dd ?    ;seg:off to call XMS driver
xms_emb dw ?     ;our EMB handle alloc'ed from XMS

cpu db ?         ;detected CPU  (3=386 4=486 5=586)

align 4
prg_base16 dd ?                        ;segs16 * 10h
prg_base32 dd ?                        ;segs32 * 10h

old_int15h dd ?                        ;old rmode int 15h handler
old_int21h dd ?                        ;old rmode int 21h handler

int3vector dd offset ints+3
           dw selcode16

picslave  db 70h                       ; PIC slave base interrupt
picmaster db 8h                        ; PIC master base interrupt

psp dw ?                               ;program segment prefix

;-----------------------------------------------------------------------------
;messages
;-----------------------------------------------------------------------------
msg_welcome db 'DOS extender Tutorial : Stage #2',13,10,'$'
msg_xms_low db 'Insufficent XMS memory found!',13,10,'$'
msg_xms_lock db 'Unable to lock XMS memory block!',13,10,'$'
msg_80386 db '80386 or better required!',13,10,'$'
msg_v86 db 'Another PMODE software is already loaded!',13,10,'$'
msg_clean_boot db 'Please use a clean boot to use this tutorial!',13,10,'$'
msg_a20 db 'Unable to enable A20!',13,10,'$'
msg_no_ram db 'Insufficent conventional memory!',13,10,'$'
data16seg ends

;-----------------------------------------------------------------------------
;16bit code segment
;-----------------------------------------------------------------------------
code16seg segment use16  ;start the 16bit code segment

;this is used to report errors and quit the program
exit16error:
  mov ah,9
  int 21h                              ;print error message
  mov dx,offset msg_clean_boot
  mov ah,9
  int 21h                              ;print clean boot message
  mov ax,4c00h
  int 21h                              ;exit program

;this will detect if the CPU is a 80386 or better
;Return : cl = processor type  (2=286 3=386 4=486)
detect_processor proc             ; get processor: 286, 386, 486, or 586
  xor cx,cx                       ; processor type 0 in case of exit

  pushf                           ; transfer FLAGS to BX
  pop bx

  mov ax,bx                       ; try to clear high 4 bits of FLAGS
  and ah,0fh

  push ax                         ; transfer AX to FLAGS
  popf
  pushf                           ; transfer FLAGS back to AX
  pop ax

  and ah,0f0h                     ; isolate high 4 bits
  cmp ah,0f0h
  je short detect_processordone   ; if bits are set, CPU is 8086/8

  mov cl,2                        ; processor type 2 in case of exit

  or bh,0f0h                      ; try to set high 4 bits of FLAGS

  push bx                         ; transfer BX to FLAGS
  popf
  pushf                           ; transfer FLAGS to AX
  pop ax

  and ah,0f0h                     ; isolate high 4 bits
  jz short detect_processordone   ; if bits are not set, CPU is 80286

  inc cx                          ; processor type 3 in case of exit

  push eax                        ; preserve 32bit registers
  push ebx     

  pushfd                          ; transfer EFLAGS to EBX
  pop ebx

  mov eax,ebx                     ; try to flip AC bit in EFLAGS
  xor eax,40000h

  push eax                        ; transfer EAX to EFLAGS
  popfd
  pushfd                          ; transfer EFLAGS back to EAX
  pop eax

  xor eax,ebx                     ; AC bit fliped?
  jz short detect_processordone2  ; if no, CPU is 386

  inc cx                          ; processor type 4 in case of exit

  mov eax,ebx                     ; try to flip ID bit in EFLAGS
  xor eax,200000h

  push eax                        ; transfer EAX to EFLAGS
  popfd
  pushfd                          ; transfer EFLAGS back to EAX
  pop eax

  xor eax,ebx                     ; ID bit fliped?
  jz short detect_processordone2  ; if no, CPU is 486

  inc cx                          ; processor type 5, CPU is 586

detect_processordone2:
  pop ebx                         ; restore 32bit registers
  pop eax    

detect_processordone:
  ret                             ; return
detect_processor endp

int15h proc                       ; this is the new INT 15h handler
  cmp ah,88h                      ;It will report how much RAM if free minus
  jnz chain_int15h                ;what we have taken
  mov ax,cs:xms_left
  iret
chain_int15h:
  jmp cs:[old_int15h]
int15h endp

;this next PROC will enable the A20 Gate so we can access RAM above 1 MB
enablea20 proc                    ; hardware enable gate A20
  pushf
  push fs
  push gs
  cli

  xor ax,ax                       ; set A20 test segments 0 and 0ffffh
  mov fs,ax
  dec ax
  mov gs,ax

  call enablea20test              ; is A20 already enabled?
  jz short enablea20done          ; if yes, done

  in al,92h                       ; PS/2 A20 enable
  or al,2
  jmp short $+2
  jmp short $+2
  jmp short $+2
  out 92h,al

  call enablea20test              ; is A20 enabled?
  jz short enablea20done          ; if yes, done

  call enablea20kbwait            ; AT A20 enable
  jnz short enablea20f0

  mov al,0d1h
  out 64h,al

  call enablea20kbwait
  jnz short enablea20f0

  mov al,0dfh
  out 60h,al

  call enablea20kbwait

enablea20f0:                      ; wait for A20 to enable
  mov cx,800h                     ; do 800h tries

enablea20l0:
  call enablea20test              ; is A20 enabled?
  jz enablea20done                ; if yes, done

  in al,40h                       ; get current tick counter
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,40h
  mov ah,al

enablea20l1:                      ; wait a single tick
  in al,40h
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,40h
  cmp al,ah
  je enablea20l1

  dec cx
  jnz enablea20l0                 ; loop for another try

  mov ax,ERROR                    ; error, A20 did not enable
  ret

enablea20done:
  pop gs
  pop fs
  popf
  xor ax,ax
  ret

enablea20kbwait:                  ; wait for safe to write to 8042
  xor cx,cx
enablea20kbwaitl0:
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,64h                       ; read 8042 status
  test al,2                       ; buffer full?
  loopnz enablea20kbwaitl0        ; if yes, loop
  ret

enablea20test:                    ; test for enabled A20
  mov al,fs:[0]                   ; get byte from 0:0
  mov ah,al                       ; preserve old byte
  not al                          ; modify byte
  xchg al,gs:[10h]                ; put modified byte to 0ffffh:10h
  cmp ah,fs:[0]                   ; set zero if byte at 0:0 not modified
  mov gs:[10h],al                 ; put back old byte at 0ffffh:10h
  ret                             ; return, zero if A20 enabled
enablea20 endp

start16 proc                      ;this is where our program starts

  push es                         ;save the PSP

  cld

  mov ax,cs
  mov ds,ax         
  mov es,ax                       ;set DS=ES=CS

  pop psp                         ;load PSP from stack

  mov dx,offset msg_welcome       ;-+
  mov ah,9                        ; |
  int 21h                         ;print Welcome message

;determine if CPU is a 80386 or better
  call detect_processor
  cmp cl,3
  jae ok_386
  mov dx,offset msg_80386
  jmp exit16error                 ;if CPU is under 386 then quit
ok_386:
  mov cpu,cl  ;save for later

;determine if we are in V86 mode (if yes then most likely EMM386,QEMM or
; Windoze is loaded so we can not continue)
  pushfd   ;push 32bit flags
  pop eax  ;pop into eax
  test eax,20000h   ;check bit 17 (VM)
  jz ok_v86
  mov dx,offset msg_v86
  jmp exit16error
ok_v86:

;determine if XMS is loaded
  mov ax,4300h
  int 2fh
  cmp al,80h
  jnz XMS_not_installed
  jmp XMS_installed
XMS_not_installed:

;alloc memory from INT 15h
  mov ah,88h
  int 15h      ;get total XMS 1k blocks free
  cmp ax,xms_min
  jae r1
  mov dx,offset msg_xms_low
  jmp exit16error
r1:   ;ok there is enough RAM
  mov cx,ax
  sub ax,xms_max
  jnc r2
  mov ax,0    ;leave no memory
r2:
  mov xms_left,ax
  sub cx,ax          ;size of our XMS RAM
  xor ebx,ebx
  mov bx,ax
  mov eax,1024*1024  ;1MB
  shl ebx,10         ;*1k
  add eax,ebx        ;eax = 1MB + XMS_left * 1k  => XMS_base
  mov xms_base,eax
  xor eax,eax
  mov ax,cx
  shl eax,10         ;*1k
  mov xms_size,eax   ;eax = # of 1k blocks * 1k => XMS_size

;Install our own int 15h handler
  mov ax,3515h
  int 21h            ;get int 15h
  mov wptr[old_int15h+2],es
  mov wptr[old_int15h+0],bx
  push ds
  pop es

  mov ax,2515h
  mov dx,offset int15h
  int 21h            ;set int 15h

;enable the a20 so we can access RAM above 1MB.
  call enablea20
  cmp ax,ERROR
  jnz r3
  mov dx,offset msg_a20
  jmp exit16error
r3:
  jmp cont_init
;-----------------------------------------------------------------------------
XMS_installed:
  mov mode,MODE_XMS
;call a non-desctructive DOS func (to avoid bugs in some XMS drivers)
  mov ah,2ch
  int 21h                              ; get time (but this is ignored)
  mov ax,4310h
  int 2fh                              ; get XMS entry-point
  mov wptr [xms_call+2],es
  mov wptr [xms_call],bx
  push cs                              ;-+
  pop es                               ;restore ES

;enable A20 thru XMS driver
  mov ah,3
  call [xms_call]                      ; global enable A20
  cmp ax,1
  jz x1
  mov dx,offset msg_a20
  jmp exit16error
x1:
  mov ah,8
  call [xms_call]                      ; query free extended memory
  ;ax=RAM free (1k blocks)  (max=64 MBs)
  cmp ax,xms_min
  jae x2
xe1:
  mov dx,offset msg_xms_low
  jmp exit16error
x2:
  cmp ax,xms_max
  jb x3
  mov ax,xms_max
x3:
  xor edx,edx
  mov dx,ax
  shl edx,10                           ;-+   ;*1k
  mov xms_size,edx                     ; |
  shr edx,10                           ;find total size of block
  mov ah,9
  call [xms_call]                      ; alloc EMB
  cmp ax,1
  jnz xe1
  mov xms_emb,dx
  mov ah,0ch
  call [xms_call]                      ; lock EMB
  cmp ax,1
  jz x4
  mov dx,xms_emb
  mov ah,0ah
  call [xms_call]                      ; free EMB
  mov dx,offset msg_xms_lock
  jmp exit16error
x4:
  shl edx,16
  mov dx,bx
  mov xms_base,edx
  
cont_init:
;setup some other variables
  xor eax,eax
  mov ax,cs
  shl eax,4                            ; convert to linear addr
  mov prg_base16,eax

  xor eax,eax
  mov ax,segs32
  shl eax,4                            ; convert to linear addr
  mov prg_base32,eax

;setup all our descriptors
  mov eax,prg_base16
  mov code16.base_lo,ax
  mov data16.base_lo,ax
  shr eax,16
  mov code16.base_mid,al
  mov data16.base_mid,al
;  mov code16.base_hi,ah               ;not needed since AH will always be 0
;  mov data16.base_hi,ah               ; "

  mov eax,prg_base32
  mov code32.base_lo,ax
  mov data32.base_lo,ax
  shr eax,16
  mov code32.base_mid,al
  mov data32.base_mid,al
;  mov code32.base_hi,ah               ; "
;  mov data32.base_hi,ah               ; "

;setup GDT and IDT base
  mov eax,prg_base16
  add eax,gdt_start
  mov gdt_addr,eax
  mov eax,prg_base16
  add eax,idt_start
  mov idt_addr,eax

;find end of RAM after EXE
  xor eax,eax
  xor ebx,ebx
  mov bx,segs32
  mov ax,TOS32
  shr ax,4
  add bx,ax                          ;bx => end of program in RAM
  shl ax,4
  cmp ax,TOS32
  jz s1
  inc bx
s1:
;setup pm_2_rm stacks
  mov pm2rm_stkend,bx
  mov ax,pm2rm_stks
  mul pm2rm_stklen
  add bx,ax
  mov pm2rm_stkcur,bx
;setup rm_2_pm stacks
  mov rm2pm_stkend,bx
  mov ax,rm2pm_stks
  mul rm2pm_stklen
  add bx,ax
  mov rm2pm_stkcur,bx
  cmp bx,0a000h
  jb s2
  mov dx,offset msg_no_ram
  jmp exit16error
s2:

;free unneeded RAM
  mov cx,psp
  mov es,cx
  sub bx,cx
  mov ah,4ah
  int 21h                              ;resize memory block

;setup IDT
  mov di,offset idt
  mov si,offset ints
  xor cx,cx
i1:
  mov [di],si
  add di,8
  inc si
  inc cl
  jnz i1
;setup other INTs
  mov wptr[idt+3*8],offset int3h
  mov wptr[idt+31h*8],offset int31h

;install our INT 21h handler (rmode)
  mov ax,3521h
  int 21h            ;get int 21h
  mov wptr[old_int21h+2],es
  mov wptr[old_int21h+0],bx
  push ds
  pop es

  mov ax,2521h
  mov dx,offset int21h
  int 21h            ;set int 21h

  cli   ;no more IRQs allowed past this point

;setup our rmode IRQ 2 PMODE redir's
  xor bx,bx
  mov bl,picmaster
  shl bx,2
  xor dx,dx
  mov dl,picslave
  shl dx,2

  xor ax,ax
  mov ds,ax
  mov si,bx  ;picmaster
  mov di,offset old_irqs
  mov cx,8
  rep movsd
  mov si,dx  ;picslave
  mov cx,8
  rep movsd
  push es                              ;-+
  push ds                              ; |
  pop es                               ; |
  pop ds                               ;swap DS and ES
  mov si,offset irq_table
  mov ax,segs16
  mov di,bx  ;picmaster
  mov cl,8
rd1:
  movsw
  stosw
  dec cl
  jnz rd1
  mov di,dx  ;picslave
  mov cl,8
rd2:
  movsw
  stosw
  dec cl
  jnz rd2

  push ds    ;-+
  pop es     ;reset ES

;we are now ready to move to 16bit PMODE

  lidt [idtr]
  lgdt [gdtr]

;clear NT and IOPL
  pushf
  mov bp,sp
  and wptr [bp+1],08fh ;40h = NT bit , 30h = IOPL bits
  popf 

;goto PMODE!
  mov eax,cr0
  or al,1           ;set PM bit
  mov cr0,eax       ;we are now in 16bit PMODE
  db 0eah           ; JMP FAR PTR SELCODE:$+4
  dw $+4,selcode16  ;  (clear prefetch que)

  mov ax,seldata16  ;reload all values
  mov ds,ax
  mov fs,ax
  mov gs,ax
  mov es,ax

  mov ss,ax
  mov sp,TOS16

;load our LDT (which is empty)  (this is not necessary)

  mov ax,selldt
  lldt ax

;setup a IRETD that will jump into our 32bit segment

  pushfd
  push dptr selcode32
  push dptr offset start32
  iretd   ;goto 32bit segment

start16 endp

;-----------------------------------------------------------------------------
; Rmode IRQ redirectors

irq0:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0001h
  jmp rm_irq_redir

irq1:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0102h
  jmp rm_irq_redir

irq2:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0203h
  jmp rm_irq_redir

irq3:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0304h
  jmp rm_irq_redir

irq4:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0405h
  jmp rm_irq_redir

irq5:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0506h
  jmp rm_irq_redir

irq6:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0607h
  jmp rm_irq_redir

irq7:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0708h
  jmp rm_irq_redir

irq8:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0809h
  jmp rm_irq_redir

irq9:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,090ah
  jmp rm_irq_redir

irqa:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0a0bh
  jmp rm_irq_redir

irqb:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0b0ch
  jmp rm_irq_redir

irqc:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0c0dh
  jmp rm_irq_redir

irqd:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0d0eh
  jmp rm_irq_redir

irqe:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0e0fh
  jmp rm_irq_redir

irqf:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0f10h
  jmp rm_irq_redir

rm_irq_redir proc
  push eax
  mov ax,cs:irqset
r1:
  rcr ax,1
  dec cl
  jnz r1
  jc doit

;branch onto normal IRQ handler
  push bx
  mov bx,offset old_irqs
  xor ax,ax
  mov al,ch
  shl ax,2
  add bx,ax
  mov eax,cs:[bx]
  mov [esp+8],eax
  pop bx
  pop eax
  pop cx
  retf

;redir to PMODE
doit:
  mov cs:rm_irqnum,ch
  cmp ch,8
  jb low_irq
;high_irq
  mov al,cs:picslave
  sub al,8
  jmp r2
low_irq:
  mov al,cs:picmaster
r2:
  add cs:rm_irqnum,al
  pop eax
  pop cx
  add esp,4                            ;unreserve space for RETF
  pushad                               ;-+
  push ds                              ; |
  push es                              ; |
  push fs                              ; |
  push gs                              ; Preserve all regs

  mov ax,segs16
  mov ds,ax

  xor ebp,ebp     ;-+
  mov bp,ss       ; |
  shl ebp,4       ; |
  add ebp,esp     ;ebp => TOS
  mov ax,sp
  push ss         ;-+
  push ax         ;Save current stack pos

  mov ax,selzero32
  mov cx,ax
;alloc a RM 2 PM stack
  mov ebx,rm2pm_stkcur                 ; EBX = ESP (from 0)
  mov edx,rm2pm_stklen
  cmp ebx,rm2pm_stkend
  jz critical_error
  sub rm2pm_stkcur,edx
  mov dx,ax   ;selzero32
  mov si,selcode16
  mov edi,offset doit2

  jmp cs:rm_2_pm

doit2:   ;now in PMODE
  db 0cdh
rm_irqnum db ?

  mov dx,ds:[ebp-2]
  mov bx,ds:[ebp-4]
  mov si,segs16
  mov di,offset doit3

  db 66h
  jmp wptr cs:pm_2_rm

doit3:
;release stack used
  mov eax,cs:rm2pm_stklen
  add cs:rm2pm_stkcur,eax
;pop all values off and end IRQ
  pop gs
  pop fs
  pop es
  pop ds
  popad
  iret
rm_irq_redir endp


;-----------------------------------------------------------------------------
;int 31h funcs 300h,301h,302h
_300h proc
  ;300h = simulate rmode INT
  ;301h = call CS:IP with RETF stack frame
  ;302h = call CS:IP with IRET stack frame
  ; in : ES:(E)DI => callstruct
  ;    : BH = 0
  ;    : BL = int # (if 300h)
  ;    : CX = # words to copy to stack
  ; out: ES:(E)DI => updated callstruct

;if you are going to trace this code I suggest you grab paper and put the
;rmode and pmode stacks on it and create them as you look thru this

  std                                  ;copy backwards
  pushad
  mov ebp,esp
  add ebp,11*4                         ;ebp -> stack params (pushad:EIP:CS:flags)
  push ds
  push es
  push fs
  push gs

  mov ds,cs:_seldata16
  mov ebx,edi
  cmp dptr es:[edi].callstruct._sp,0
  jz s1
;get SS:SP from callstruct
  xor eax,eax
  mov ax,es:[edi].callstruct._ss
  push ax                              ;save for later (rmode SS)
  shl eax,4
  xor edx,edx
  mov dx,es:[edi].callstruct._sp
  push dx                              ;save for later (rmode SP)
  add eax,edx
  jmp c1                               ; EAX = rmode TOS, DX = SP
s1:  ;supply system stack
  xor eax,eax
  mov ax,pm2rm_stkcur                  ;get next stack location
  cmp ax,pm2rm_stkend                  ;check if a stack is avail.
  jnz ok1
  jmp int31hfail
ok1:
  mov dx,pm2rm_stklen                  ;make stack pt to next location
  sub ax,dx
  sub pm2rm_stkcur,dx
  push ax                              ;save for later (rmode SS)
  add ax,dx
  shl eax,4
  shl dx,4                             ; EAX = rmode TOS, DX = SP
  push dx                              ;save for later (rmode SP)
c1:
;preserve current rmode stack ptr (SP) (so it will be easy to pop off
; parms from rmode stack later)
  sub dx,6                             ;reserve space for SS:ESP on rmode stk
  xchg dx,old_sp
  push dx                              ;preserve old_sp
;prepare rmode stack ptr
  push es                              ;preserve ES
  mov es,_selzero32                    ;-+
  mov edi,eax                          ;es:edi => dest stack
;copy SS:ESP onto rmode stack so they can be restored later on
  sub edi,2
  mov es:[edi],ss
  sub edi,4
  lea eax,[esp+2]
  db 67h
  stosd
  add edi,2
;check if there are any stack params
  or cx,cx
  jz c2
;prepare pmode stack ptr
  push ss                              ;-+
  pop ds                               ; |
  mov esi,ebp                          ;ds:esi => src stack
;copy over CX words from current stack to new stack (params)
  xor edx,edx
  mov dx,cx
  mov ecx,edx                          ;clear high word of ecx
  dec dx
  shl edx,1
  add esi,edx
  db 67h                               ;-+
  rep movsw                            ;copy stack over
  add esi,2                            ;move back to normal pos
;copy (maybe Flags):CS:IP to rmode stack (used to return back to pmode)
c2:
  cmp bptr[ebp-11*4].callstruct._eax,1
  jz f4
  pushf
  pop ax
  db 67h
  stosw
f4:
  mov ax,segs16
  db 67h
  stosw 
  mov ax,offset r2
  db 67h
  stosw
;copy Flags:CS:IP to call rmode int/proc
  pop ds                               ;restore ES into DS
  mov ax,[ebx].callstruct._flg         ;flags from callstruct
  cmp bptr[ebp-11*4].callstruct._eax,1
  jz f1
  and ah,0fch                          ;clear IF and TF for func 300h,302h
f1:
  db 67h
  stosw                                ;save flags on rmode stack
  cmp bptr[ebp-11*4].callstruct._eax,0
  jnz f2
;300h
  xor eax,eax
  mov al,bptr[ebp-11*4].callstruct._ebx
  shl eax,2  ;*4
  mov eax,es:[eax]                     ;get rmode INT # CS:IP
  jmp f3
f2:
;301h and 302h
  mov eax,dptr[ebx].callstruct._ip
f3:
  sub edi,2
  db 67h
  stosd
  add edi,2
;copy FS and GS for real mode
  mov ax,[ebx].callstruct._gs
  db 67h
  stosw
  mov ax,[ebx].callstruct._fs
  db 67h
  stosw
;copy over callstruct (only what is needed)
  sub edi,2
  mov ecx,8                            ;8 dwords to copy
  mov esi,ebx                          ;ds:esi => callstruct
  add esi,7*4                          ;start from bottom of block
  db 67h                               ;-+
  rep movsd                            ;copy callstruct over
  add esi,4                            ;ds:esi => callstruct
  add edi,4                            ;es:edi => rmode tos

;  AX = new DS
;  CX = new ES
;  DX = new SS
;  BX = new SP
;  SI = new CS
;  DI = new IP

  xor eax,eax
  mov dx,[esp+4]                       ;SS
  mov ax,dx
  shl eax,4
  sub edi,eax
  mov bx,di                            ;SP

  mov ax,[esi].callstruct._ds          ;DS
  mov cx,[esi].callstruct._es          ;ES
  mov si,segs16                        ;CS
  mov di,offset r1                     ;IP

  db 66h                               ;-+
  jmp cs:pm_2_rm                       ;jmp DWORD PTR  (32bit offset)

r1:

  popad
  pop fs
  pop gs

  iret

r2:    ;return back to PMODE
;remove stack parameters
  mov sp,cs:old_sp
  push gs                              ;order preserved is important
  push fs
  push ds
  push es
  pushf
  pushad

  mov bp,sp
  mov ebx,[bp+42]                      ;for ESP
  mov dx,[bp+46]                       ;for SS
  mov si,selcode16
  mov ax,selzero32
  mov cx,ax
  mov edi,offset p1

;  AX  = new DS
;  CX  = new ES
;  DX  = new SS
;  EBX = new ESP
;  SI  = new CS
;  EDI = new EIP

  jmp cs:rm_2_pm

p1:
  mov ebx,cs:prg_base16
  pop ds:old_sp[ebx]

  mov edi,[esp+6*2].callstruct._edi  ;get saved EDI
  mov es,[esp+4*2]                   ;get saved ES

;copy rmode stack to callstruct
  xor eax,eax
  pop ax                               ;pop rmode SP
  xor esi,esi
  pop si                               ;pop rmode SS
  shl esi,4
  add esi,eax
  sub esi,42+6                         ;move back to saved regs on rmode stack
  mov ecx,10
  cld
  mov eax,es:[edi].callstruct._res     ;this must be reserved
  mov [esi].callstruct._res,eax
  db 67h
  rep movsd
  db 67h                               ;copy last word
  movsw
;release system stack
  cmp dptr es:[edi].callstruct._sp,0
  jz p4
  mov ax,ds:pm2rm_stklen[ebx]
  add ds:pm2rm_stkcur[ebx],ax          ;stack released
p4:
  jmp int31hok
_300h endp

;-----------------------------------------------------------------------------
;Common INT 31h return routines

int31hok:
  pop gs
  pop fs
  pop es
  pop ds
  popad
  and bptr[esp+8],0feh                 ;clear carry flag = sucessful
  iretd

int31hfail:
  pop gs
  pop fs
  pop es
  pop ds
  popad
  or bptr[esp+8],1h                    ;set flag = failed
  iretd

;-----------------------------------------------------------------------------
; int 31h func 305h  (raw/xms)

_305h proc
  ;get raw mode state save/restore addr's
;   AX     = size of buffer in bytes required to save state
;   BX:CX  = segment:offset of real mode routine used to save/restore state
;   SI:EDI = selector:offset of protected mode routine used to save/restore
;            state
  xor ax,ax
  mov bx,segs16
  mov cx,cs:sr_state_rm
  mov si,selcode16
  mov edi,cs:sr_state_pm
  and bptr[esp+11],0feh  ;clear carry
  iretd
_305h endp

_306h proc
  ;get raw mode switching addr's
;   BX:CX  = segment:offset of real to protected mode switch procedure
;   SI:EDI = selector:offset of protected to real mode switch procedure
  pop edi
  mov bx,segs16
  mov cx,cs:rm_2_pm
  mov si,selcode16
  mov edi,cs:pm_2_rm
  and bptr[esp+11],0feh  ;clear carry
  iretd
_306h endp

int31h_table_start equ $
int31h_table dw 300h,301h,302h,305h,306h
int31h_table_size equ ($-int31h_table_start)/2
int31h_offset dw offset _300h
              dw offset _300h
              dw offset _300h
              dw offset _305h
              dw offset _306h

int31h proc
  ;ax=func #
  push edi
  push ecx
  push es
  mov es,cs:_seldata16
  mov di,offset int31h_table
  mov ecx,int31h_table_size
  repnz scasw
  jnz bad
  mov edi,int31h_table_size
  sub edi,ecx
  dec edi
  shl edi,1
  add edi,offset int31h_offset
  mov di,es:[edi]
  pop es
  pop ecx
  shl edi,16         ;move into high word so we can skip over low word
  xchg edi,[esp]     ;pop edi, store RET addr
  add esp,2          ;skip over low word (it's only a 16bit addr)
  ret                ;jmp to function handler
bad:
  pop es
  pop ecx
  pop edi
  ; EIP , CS , EFlags
  or bptr[esp+8],1  ;set carry
  iretd
int31h endp

;-----------------------------------------------------------------------------
; pm 2 rm & rm 2 pm   switching proc for RAW/XMS
; save/restore procs

pm_2_rm_rx proc  ;switches to Rmode  (raw/xms)

;  AX = new DS
;  CX = new ES
;  DX = new SS
;  BX = new SP
;  SI = new CS
;  DI = new IP

  push ax   ;save AX
  mov ds,cs:_selzero32
  xor eax,eax
  mov ax,dx
  shl eax,4
  movzx ebx,bx
  add eax,ebx
  pushf
  cli
  pop ds:[eax-2]
  mov ds:[eax-4],si
  mov ds:[eax-6],di
  pop di    ;pop ax into di
  lidt cs:rm_idtr
  mov ax,seldata16
  mov ds,ax
  mov es,ax
  mov fs,ax
  mov gs,ax
  mov ss,ax
  sub ebx,6   ;make room for iret data
  mov esp,ebx
  mov eax,cr0
  and al,0feh
  mov cr0,eax     ;now in rmode
  db 0eah
  dw $+4,segs16
  mov ss,dx
  mov ds,di       ;ax was poped into di
  mov es,cx
  xor ax,ax
  mov fs,ax
  mov gs,ax
  iret            ;goto target addr in real mode
pm_2_rm_rx endp

rm_2_pm_rx proc   ;switches to Pmode from Rmode

;  AX  = new DS
;  CX  = new ES
;  DX  = new SS
;  EBX = new ESP
;  SI  = new CS
;  EDI = new EIP

  pushfd
  cli
  push ax
  lgdt cs:gdtr
  lidt cs:idtr
  mov eax,cr0
  or al,1
  mov cr0,eax
  db 0eah
  dw $+4,selcode16
  pop ds
  mov es,cx
  xor ax,ax
  mov fs,ax
  mov gs,ax
  pop eax
  mov ss,dx                       ; load protected mode SS:ESP
  mov esp,ebx
  and ah,0bfh                     ; set NT=0 in old EFLAGS
  push ax                         ; set current FLAGS
  popf
  push eax                        ; store old EFLAGS
  push esi                        ; store protected mode target CS
  push edi                        ; store protected mode target EIP
  iretd                           ; go to targed addx in protected mode
rm_2_pm_rx endp

sr_state_pm_rx:
  db 66h     ;32bit retf
sr_state_rm_rx:
  retf   ;nothing to save/restore

;this proc was copied almost directly from PMODE v3.07 - why reinvent the
;wheel?
int3h proc ;pmode
  ;this will redir INTs (and IRQs) to rmode
  cmp dptr[esp],offset ints            ;I added these extra checks in case
  jb do3                               ;this kernel is debugged later
  cmp dptr[esp],offset ints+255
  ja do3
  cmp wptr[esp+4],selcode16            ;was it from our ints table or is it
  jz redir                             ; an embedded int 3?
;continue on to normal INT 3 handler
do3:
  jmp fptr cs:int3vector               ; JMP FAR sel:off32 handler
;redirect to proper handler in rmode
redir:
  mov [esp+8],eax                      ; store EAX for later POPAD
  mov eax,[esp]                        ; get address in redirection matrix
  add esp,8                            ; discard EIP and CS from INT 3
  push ecx                             ;-+
  push edx                             ; |
  push ebx                             ; |
  push esp                             ; |
  push ebp                             ; |
  push esi                             ; |
  push edi                             ;store rest of registers for POPAD
  push ds                              ;-+
  push es                              ; |
  push fs                              ; |
  push gs                              ;store all segment registers

  mov ds,cs:_selzero32                 ; DS -> 0 (beginning of memory)
  mov edi,cs:prg_base16                ; EDI = offset of SEGS16 from 0

  mov dx,cs:pm2rm_stkcur               ; DX = SS for real mode redirection
  xor ebp,ebp                          ;-+
  mov bp,dx                            ; |
  shl ebp,4                            ; EBP -> top of real mode stack

  cmp dx,cs:pm2rm_stkend               ; exceeded real mode stack space?
  jz critical_error                    ; if yes, critical error (hang)

  mov bx,cs:pm2rm_stklen               ; get size of real mode stack
  sub dx,bx                            ; adjust DX to next stack location
  sub ds:pm2rm_stkcur[edi],bx          ; update ptr for possible reenterancy
  shl bx,4                             ; set real mode SP to top of stack

  mov ds:[ebp-2],ss                    ; store SS:ESP on real mode stack
  mov ds:[ebp-6],esp

  sub ax,offset ints+1                 ; AX = int number
                                       ; the +1 is because the next instruct
                                       ; to return to is after the one-byte
                                       ; int 3 opcode
  mov ah,al                            ;-+
  and ah,0f8h                          ; AH = high 5 bits of int number

  cmp ah,cs:picmaster                  ; low IRQ?
  jz redir_irq                         ; if yes, do IRQ
  cmp ah,cs:picslave                   ; high IRQ?
  jnz redir_int                        ; if no, do INT (with general regs)
  add al,8                             ; upper 8 IRQs
;-----------------------------------------------------------------------------
redir_irq:                             ; an IRQ redirection
  mov si,segs16                        ; real mode target CS:IP
  mov di,offset r1       
  sub bx,6                             ; adjust real mode SP for stored vars
  sub al,ah                            ; convert INT # to IRQ #
  xor ah,ah
  shl ax,2
  mov bp,offset old_irqs
  add bp,ax
  mov ax,segs16                        ; rmode DS

  db 66h                               ; JMP DWORD PTR, as in 32bit offset,
  jmp wptr cs:pm_2_rm                  ;  not seg:16bit offset

r1:
  pushf
  push cs
  push offset r1b
  push dptr ds:[bp]
  retf
r1b:
  mov ax,seldata16                     ; DS selector value for protected mode
  mov cx,ax                            ; ES selector value for protected mode
  pop ebx                              ; get protected mode SS:ESP from stack
  pop dx
  mov si,selcode16                     ; target CS:EIP in protected mode
  mov edi,offset r2

  jmp cs:rm_2_pm                       ; go back to protected mode

r2:
  mov ax,pm2rm_stklen                  ; release real mode stack
  add pm2rm_stkcur,ax

  pop gs                               ;-+
  pop fs                               ; |
  pop es                               ; |
  pop ds                               ; |
  popad                                ;restore all registers
  iretd

;-----------------------------------------------------------------------------
redir_int:                             ; an INT redirection
  mov ds:intnum[edi],al                ; modify code with interrupt number

  mov es,cs:_selzero32                 ; copy registers from protected mode
  lea edi,[ebp-26h]                    ;  stack to real mode stack
  lea esi,[esp+8]                      ; esp+8 => popad
  mov ecx,8
  cld
  rep movsd es:[edi],ss:[esi]          ; note the SS override!

  mov si,segs16                        ; real mode target CS:IP
  mov di,offset i1
  sub bx,26h                           ; adjust real mode SP for stored vars

  db 66h                               ; JMP DWORD PTR, as in 32bit offset,
  jmp wptr cs:pm_2_rm                  ;  not seg:16bit offset

i1:
  popad                                ; load regs with int call values

  db 0cdh                              ; INT intnum
intnum db ?

  pushad                               ; store registers on stack
  pushf                                ; store flags on stack
  cli
  cld

  xor eax,eax                          ; EAX = linear ptr to SS
  mov ax,ss
  shl eax,4
  xor ebp,ebp                          ;-+
  mov bp,sp                            ; EBP = SP

  mov ebx,[bp+22h]                     ; get protected mode SS:ESP from stack
  mov dx,[bp+26h]

  add ebp,eax                          ; EBP -> stored regs on stack

  mov ax,selzero32                     ; DS selector value for pmode zero32
  mov cx,dx                            ; ES selector value for pmode stack
  mov si,selcode16                     ; target CS:EIP in protected mode
  mov edi,offset i2

  jmp cs:rm_2_pm                       ; go back to protected mode

i2:
  mov ax,ds:[ebp]                      ; move return FLAGS from real mode
  and ax,8d5h                          ;  stack to protected mode stack
  mov bx,[esp+30h]                     ; get flags from stack
  and bx,not 8d5h
  or ax,bx                             ; merge pmode/rmode flags
  mov [esp+30h],ax                     ; put flags onto IRETD

  lea esi,[ebp+2]                      ;DS:ESI => rmode pushad
  lea edi,[esp+8]                      ;ES:EDI => pmode popad
  mov ecx,8
  db 67h                               ;-+
  rep movsd                            ; copy from rmode stack to pmode stack

  mov edi,cs:prg_base16                ; release  real mode stack
  mov ax,cs:pm2rm_stklen
  add ds:pm2rm_stkcur[edi],ax

  pop gs                               ;-+
  pop fs                               ; |
  pop es                               ; |
  pop ds                               ; |
  popad                                ; restore all regs
  iretd
int3h endp

critical_error proc                    ; some unrecoverable error
  cli                                  ; make sure we are not interrupted
  in al,61h                            ; beep
  or al,3
  out 61h,al
  jmp $                                ; now hang
critical_error endp

int21h proc  ;INT 21h handler (rmode)
  cmp ah,4ch                           ; is program terminating?
  jz exit
normal:
  jmp cs:old_int21h                    ;no, continue on to old int 21h handler
exit:                                  ;yes, program is terminating
  cmp wptr[esp+2],segs16               ;-+
  jnz normal                           ;make sure it's from our program
  push eax
;restore all RMODE IRQs
  xor ax,ax
  mov es,ax
  mov ax,segs16
  mov ds,ax
  xor bx,bx
  mov bl,picmaster
  shl bx,2
  xor dx,dx
  mov dl,picslave
  shl dx,2
  mov di,bx  ;picmaster
  mov si,offset old_irqs
  mov cx,8
  rep movsd
  mov di,dx  ;picslave
  mov cx,8
  rep movsd

  mov eax,old_int21h
  mov es:[21h*4],eax                   ;restore old INT 21h handler
  cmp mode,MODE_XMS
  jz exit_xms                          ; jmp to XMS clean-up

  mov eax,old_int15h                   ;INT 15h clean-up
  mov es:[15h*4],eax                   ;restore old INT 15h handler
cont:
  pop eax
  jmp old_int21h

exit_xms:                              ; XMS clean-up
  mov ah,0dh
  mov dx,xms_emb
  call [xms_call]                      ; unlock RAM
  mov ah,0ah
  mov dx,xms_emb
  call [xms_call]                      ; free RAM used
  jmp cont
int21h endp

code16seg ends

assume cs:segs32,ds:segs32,ss:segs32,es:NOTHING

data32seg segment use32
  x dd 0     ;coords of cursor on screen
  y dd 0
  timer dw 0

;32bit messages
  msg32_timer db ' = TIMER',13,0
  msg32_welcome db 'Welcome to 32bit PMODE!',13,13,0
  msg32_kbd db 'Test out the keyboard, press ESC to quit!',13,0
  key db ?,0
  msg32_test db 'Testing int 31h ax=300h$'
  cb callstruct <>

;define selectors within data32seg
  __selcode16  dw selcode16
  __seldata16  dw seldata16
  __selvid     dw selvid
  __seldata32  dw seldata32
  __selzero32  dw selzero32
  __selcode32  dw selcode32
  __selldt     dw selldt

data32seg ends

code32seg segment use32

start32 proc
;reload all segment regs with 32bit selectors
  mov ax,seldata32
  mov ds,ax
  mov es,ax
  mov ss,ax
  mov fs,ax
  mov gs,ax
  mov esp,TOS32
  sti  ;enable IRQs

  call clrscr   ;clear the screen

  mov esi,offset msg32_timer
  call print

  mov esi,offset msg32_welcome
  call print

  mov esi,offset msg32_kbd
  call print

  mov ax,selcode32
  mov edx,offset timer_irq
  mov bl,8
  call setint

  mov x,0
  mov y,2
  call gotoxy
;test DPMI func #300h
  mov bptr cb._eax+1,9h                ;ah=9
  mov cb._ds,segs32
  mov cb._edx,offset msg32_test        ;high part of EDX must not be used
  mov cb._ss,0                         ;-+
  mov cb._sp,0                         ;SS:SP must be zero in order to use
                                       ; system stacks
  mov edi,offset cb
  mov bx,21h                           ; BH must be 0
  mov ax,300h
  xor cx,cx
  int 31h

  mov x,0
  mov y,5
  call gotoxy
g1:
  mov ax,0      ;now we can sit in rmode and wait for a key and
  int 16h       ; IRQ#0 will still come upto PMODE

  cmp ah,1      ;is it the ESC scan code?
  jz quit       ;yes then quit
  mov key,al
  mov esi,offset key
  call print
  jmp g1

quit:
  mov ax,4c00h
  int 21h  ;quit program

start32 endp

clrscr proc
  mov es,__selvid
  xor edi,edi         ;linear addr of 0b800:0  (base=0b8000h)
  mov ecx,80*25
  mov ax,720h         ;07h=white on black 20h=spaces
  rep stosw
  mov x,0
  mov y,0
  mov es,__seldata32
  ret
clrscr endp

;esi = string to print
print proc
  mov es,__selvid
p0:
  mov edi,y
  imul edi,edi,80*2    ;80*2 = # bytes/row (in text mode)d
  add edi,x
  add edi,x            ;add twice to skip over color bytes too
p1:
  cmp bptr[esi],0
  jz p3
  cmp bptr[esi],13
  jz p13
  movsb                ;ds:esi => es:edi
  inc edi              ;skip over color byte
  inc x
  cmp x,80
  jnz p1
p2:                    ;skip to next line
  mov x,0
  inc y
  cmp y,25
  jnz p0
  mov y,0
  jmp p0
p13:
  inc esi
  jmp p2
p3:
  mov es,__seldata32
  call gotoxy
  ret
print endp

timer_irq proc
  ;this is carled during IRQ#0
  push eax
  push es
  push ds
  mov es,cs:_selvid
  mov ds,cs:_seldata32
  mov ax,timer
  inc timer
  mov bptr es:[0],al
  mov al,20h
  out 20h,al                           ;ack IRQ
  pop ds
  pop es
  pop eax
  iretd
timer_irq endp

setint proc
  ;sets one of the exception handlers to a new 32bit location
  ;by default all INTs goto rmode
  ;in : AX=selector
  ;   :EDX=offset
  ;   : BL=INT #  (currently only 0-31 are allowed - we will make our IDT
  ;                full size -256 vektorz- in the next tutorial) 
  cli  ;disable IRQs while setting INT vektor
  movzx ebx,bl
 
  mov cx,seldata16
  mov ds,cx
  mov esi,offset idt
  mov [esi+ebx*8],dx                   ; set the offset (0-15)
  mov [esi+ebx*8+2],ax                 ; set the selector
  shr edx,16
  mov [esi+ebx*8+6],dx                 ; set the offset (16-31)

assume ds:segs16
  mov ds,__seldata16
  mov dl,bl
  and dl,0f8h  ;mask high 5 bits
  cmp dl,picmaster
  jnz i1
  mov dl,picmaster
  jmp s1
i1:
  cmp dl,picslave
  jnz i2
  mov dl,picslave
  add dl,8
  jmp s1
i2:
  mov ds,_seldata32
  sti  ;reenable IRQs
  ret
s1:
  xor dh,dh
  sub bx,dx
  mov cl,bl
  mov dx,1
  shl dx,cl
  or irqset,dx
  jmp i2  
assume ds:segs32
setint endp

gotoxy proc     ;pos cursor at (x,y)
  mov ax,200h
  mov dl,byte ptr[x]
  mov bh,0
  mov dh,byte ptr[y]
  int 10h
  ret
gotoxy endp

code32seg ends

end start16     ;start program in segs16:start16

