data16seg segment

msg_a20 db 'Unable to enable A20!',13,10,'$'

int15h_left dw ?    ;int 15h will report this much is free
int15h_old dd ?     ;old rmode int 15h handler

old_sp dw ?              ;used by 30xh DPMI funcs

int3vector dd offset ints+3
           dw selcode16

;define our intrmatrix
ints db 256 dup (0cch)                 ;this is what each of the default
                                       ;INT descriptors pts to
                                       ;0cch = int 3 (one-byte opcode)
                                       ;so ALL INTs goto the INT 3 handler
                                       ;which will direct the INT to rmode

;define other variables needed
irq_table dw offset irq0
          dw offset irq1
          dw offset irq2
          dw offset irq3
          dw offset irq4
          dw offset irq5
          dw offset irq6
          dw offset irq7
          dw offset irq8
          dw offset irq9
          dw offset irqa
          dw offset irqb
          dw offset irqc
          dw offset irqd
          dw offset irqe
          dw offset irqf

old_irqs dd 16 dup (?)   ;old rmode IRQ handlers
irqset dw 0              ;each bit is set if an IRQ handler is setup in PMODE
                         ;for that IRQ

data16seg ends

code16seg segment

chk_v86 proc
;determine if we are in V86 mode (if yes then most likely EMM386,QEMM or
; Windoze is loaded so we can not continue)
  smsw ax                              ; AX = machine status word
  test al,1                            ; is system in protected mode?
  jnz inv86                            ; if in protected mode then exit
  ret
inv86:
  mov dx,offset msg_v86
  jmp exit16error
chk_v86 endp

raw_init proc
;alloc memory from INT 15h
  mov ah,88h
  int 15h      ;get total extended 1k blocks free
  mov cx,ax
  mov ebx,ram_max
  shr ebx,10   ;convert bytes to Ks
  .if ebx>=64*1024   ;64MBs?
    xor ax,ax
  .else
    sub ax,bx
    .if carry?
      xor ax,ax    ;leave no memory
    .endif
  .endif
  mov int15h_left,ax
  sub cx,ax          ;size of our XMS RAM
  xor ebx,ebx
  mov bx,ax
  mov eax,1024*1024  ;1MB
  shl ebx,10         ;*1k
  add eax,ebx        ;eax = 1MB + int15h_left * 1k  => XMS_base
  mov xms_base,eax
  xor eax,eax
  mov ax,cx
  shl eax,10         ;*1k
  mov xms_size,eax   ;eax = # of 1k blocks * 1k => XMS_size

;Install our own int 15h handler
  mov ax,3515h
  int 21h            ;get int 15h
  mov wptr[int15h_old+2],es
  mov wptr[int15h_old+0],bx
  push ds
  pop es

  mov ah,25h
  mov dx,offset int15h
  int 21h            ;set int 15h

  mov inited,1       ;after this point raw_uninit should be called on fatal
                     ;errors

;enable the a20 so we can access RAM above 1MB.
  call enablea20
  cmp ax,ERROR
  jnz r3
  mov dx,offset msg_a20
  jmp exit16error
r3:
  ret
raw_init endp

raw_uninit proc
; in : ds=segs16  es=0
  mov eax,int15h_old                   ;INT 15h clean-up
  mov es:[15h*4],eax                   ;restore old INT 15h handler
  ret
raw_uninit endp

int15h proc                       ; this is the new INT 15h handler
  cmp ah,88h                      ;It will report how much RAM if free minus
  jnz chain_int15h                ;what we have taken
  mov ax,cs:int15h_left
  iret
chain_int15h:
  jmp cs:[int15h_old]
int15h endp

;this next PROC will enable the A20 Gate so we can access RAM above 1 MB
enablea20 proc                    ; hardware enable gate A20
  pushf
  push fs
  push gs
  cli

  xor ax,ax                       ; set A20 test segments 0 and 0ffffh
  mov fs,ax
  dec ax
  mov gs,ax

  call enablea20test              ; is A20 already enabled?
  jz short enablea20done          ; if yes, done

  in al,92h                       ; PS/2 A20 enable
  or al,2
  jmp short $+2
  jmp short $+2
  jmp short $+2
  out 92h,al

  call enablea20test              ; is A20 enabled?
  jz short enablea20done          ; if yes, done

  call enablea20kbwait            ; AT A20 enable
  jnz short enablea20f0

  mov al,0d1h
  out 64h,al

  call enablea20kbwait
  jnz short enablea20f0

  mov al,0dfh
  out 60h,al

  call enablea20kbwait

enablea20f0:                      ; wait for A20 to enable
  mov cx,800h                     ; do 800h tries

enablea20l0:
  call enablea20test              ; is A20 enabled?
  jz enablea20done                ; if yes, done

  in al,40h                       ; get current tick counter
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,40h
  mov ah,al

enablea20l1:                      ; wait a single tick
  in al,40h
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,40h
  cmp al,ah
  je enablea20l1

  dec cx
  jnz enablea20l0                 ; loop for another try

  mov ax,ERROR                    ; error, A20 did not enable
  ret

enablea20done:
  pop gs
  pop fs
  popf
  xor ax,ax
  ret

enablea20kbwait:                  ; wait for safe to write to 8042
  xor cx,cx
enablea20kbwaitl0:
  jmp short $+2
  jmp short $+2
  jmp short $+2
  in al,64h                       ; read 8042 status
  test al,2                       ; buffer full?
  loopnz enablea20kbwaitl0        ; if yes, loop
  ret

enablea20test:                    ; test for enabled A20
  mov al,fs:[0]                   ; get byte from 0:0
  mov ah,al                       ; preserve old byte
  not al                          ; modify byte
  xchg al,gs:[10h]                ; put modified byte to 0ffffh:10h
  cmp ah,fs:[0]                   ; set zero if byte at 0:0 not modified
  mov gs:[10h],al                 ; put back old byte at 0ffffh:10h
  ret                             ; return, zero if A20 enabled
enablea20 endp

;-----------------------------------------------------------------------------
; pm 2 rm & rm 2 pm   switching proc for RAW/XMS
; save/restore procs

pm_2_rm_rx proc  ;switches to Rmode  (raw/xms)

;  AX = new DS
;  CX = new ES
;  DX = new SS
;  BX = new SP
;  SI = new CS
;  DI = new IP
; EBP = preserved

  push ax   ;save AX
  mov ds,cs:_selzero32
  xor eax,eax
  mov ax,dx
  shl eax,4
  movzx ebx,bx     ;can't remove this movzx
  add eax,ebx
  pushf
  cli
  pop ds:[eax-2]
  mov ds:[eax-4],si
  mov ds:[eax-6],di
  pop di    ;pop ax into di
  lidt cs:rm_idtr
  mov ax,seldata16
  mov ds,ax
  mov es,ax
  mov fs,ax
  mov gs,ax
  mov ss,ax
  sub ebx,6   ;make room for iret data
  mov esp,ebx
  mov eax,cr0
  and al,0feh
  mov cr0,eax     ;now in rmode
  db 0eah
  dw $+4,segs16
  mov ss,dx
  mov ds,di       ;ax was poped into di
  mov es,cx
  xor ax,ax
  mov fs,ax
  mov gs,ax
  iret            ;goto target addr in real mode
pm_2_rm_rx endp

rm_2_pm_rx proc   ;switches to Pmode from Rmode

;  AX  = new DS
;  CX  = new ES
;  DX  = new SS
;  EBX = new ESP
;  SI  = new CS
;  EDI = new EIP
;  EBP = preserved

  pushfd
  cli
  push ax
  lgdt cs:gdtr
  lidt cs:idtr
  mov eax,cr0
  or al,1
  mov cr0,eax
  db 0eah
  dw $+4,selcode16
  pop ds
  mov es,cx
  xor ax,ax
  mov fs,ax
  mov gs,ax
  pop eax
  mov ss,dx                       ; load protected mode SS:ESP
  mov esp,ebx
  and ah,0bfh                     ; set NT=0 in old EFLAGS
  push ax                         ; set current FLAGS
  popf
  push eax                        ; store old EFLAGS
  push esi                        ; store protected mode target CS
  push edi                        ; store protected mode target EIP
  iretd                           ; go to targed addx in protected mode
rm_2_pm_rx endp

sr_state_pm_rx:
  db 66h     ;32bit retf
sr_state_rm_rx:
  retf   ;nothing to save/restore

init_rxv proc  ;final init for Raw/XMS/Vcpi
;new stage#8 : now the GDT size is increased if sels>0
  .if sels
    ;calc RAM needed
    xor eax,eax
    xor ebx,ebx
    mov ax,gdt_size
    mov bx,sels
    shl ebx,3
    add eax,ebx
    mov gdt_end,eax
    add eax,0fh
    shr eax,4
    .if eax>=64*1024
      mov ax,0ffffh  ;this will guaranty a fail with getRAM
    .endif
    call getRAM
    mov es,bx
    xor eax,eax
    mov ax,bx
    shl eax,4
    add eax,prg_base16
    add gdt_addr,eax
    mov eax,gdt_addr
    add gdt_end,eax
    mov si,offset null  ;1st entry in GDT
    xor di,di
    mov cx,(gdt_size)/4
    add bx,6  ;increment segment to pt to after my descriptors
    .if mode!=MODE_VCPI
      sub cx,8   ;under raw/xms we don't need the VCPI descriptors
      sub bx,2   ;dec segment to pt to right before the VCPI descriptors
    .endif
    rep movsd
;make all new selectors invalid (to prevent bugs)
    mov cx,sels
    mov eax,0   ;at least the P bit will be clear making the Descriptor invalid
    xor di,di
    mov es,bx
g1:
    stosd
    stosd
    dec cx
    jz done
    stosd
    stosd
    inc bx
    mov es,bx
    xor di,di
    dec cx
    jnz g1
done:
    push cs
    pop es
  .else
    mov eax,prg_base16
    add eax,gdt_start
    mov gdt_addr,eax
    mov gdt_end,eax   ;NO descriptors are avail.
  .endif
;setup IDT base
  mov eax,prg_base16
  add eax,idt_start
  mov idt_addr,eax

;setup pm_2_rm stacks
  mov ax,pm2rm_stks
  mul pm2rm_stklen
  mov cx,ax
  call getRAM
  mov pm2rm_stkend,bx
  add bx,cx
  mov pm2rm_stkcur,bx

;setup rm_2_pm stacks
  mov ax,rm2pm_stks
  mul wptr[rm2pm_stklen]
  shl rm2pm_stklen,4        ;NOTE : These must be converted to linear !!!
  mov cx,ax
  call getRAM
  mov wptr[rm2pm_stkend],bx
  shl rm2pm_stkend,4        ;convert to linear
  add bx,cx
  mov wptr[rm2pm_stkcur],bx
  shl rm2pm_stkcur,4        ;convert to linear

;alloc ram for CALLBACKs
  mov cl,callbacks
  or cl,cl
  jz cb2                               ; no callbacks?
  xor ch,ch
  mov ax,25                            ; 25 bytes for each callback
  mul cx
  add ax,0fh
  shr ax,4
  call getRAM
  mov callbackseg,bx
  mov wptr callbackbase,bx
  shl callbackbase,4
;setup the CALLBACKs
  mov cl,callbacks
  mov ds,callbackseg
  xor di,di
cb1:
  mov word ptr [di],6066h              ; PUSHAD instruction
  mov byte ptr [di+2],068h             ; PUSH WORD instruction
  mov word ptr [di+3],0                ; immediate 0 used as free flag
  mov word ptr [di+5],06866h           ; PUSH DWORD instruction
  mov byte ptr [di+11],0b9h            ; MOV CX,? instruction
  mov word ptr [di+14],06866h          ; PUSH DWORD instruction
  mov byte ptr [di+20],0eah            ; JMP FAR PTR ?:? intruction
  mov word ptr [di+21],offset callback
  mov word ptr [di+23],segs16
  add di,25                            ; increment ptr to next callback
  dec cl                               ; decrement loop counter
  jnz cb1                              ; if more callbacks to do, loop
  push cs
  pop ds
cb2:
;setup IDT
  mov di,offset idt
  mov si,offset ints
  xor cx,cx
i1:
  mov [di],si
  add di,8
  inc si
  inc cl
  jnz i1
;setup other INTs
  mov wptr[idt+3*8],offset int3h
  mov wptr[idt+31h*8],offset int31h
  ret
init_rxv endp

irq_init proc                          ;setup our rmode IRQ 2 PMODE redir's
  cli
  xor bx,bx
  mov bl,picmaster
  shl bx,2
  xor dx,dx
  mov dl,picslave
  shl dx,2

  xor ax,ax
  mov ds,ax
  mov si,bx  ;picmaster
  mov di,offset old_irqs
  mov cx,8
  rep movsd
  mov si,dx  ;picslave
  mov cx,8
  rep movsd

  push es                              ;-+
  push ds                              ; |
  pop es                               ; |
  pop ds                               ;swap DS and ES
  mov si,offset irq_table
  mov ax,segs16
  mov di,bx  ;picmaster
  mov cl,8
rd1:
  movsw
  stosw
  dec cl
  jnz rd1
  mov di,dx  ;picslave
  mov cl,8
rd2:
  movsw
  stosw
  dec cl
  jnz rd2

  push ds    ;-+
  pop es     ;reset ES

  sti
  ret
irq_init endp

irq_uninit proc
;restore all RMODE IRQs
; in : ds=segs16 es=0
  xor bx,bx
  mov bl,picmaster
  shl bx,2
  xor dx,dx
  mov dl,picslave
  shl dx,2
  mov di,bx  ;picmaster
  mov si,offset old_irqs
  mov cx,8
  rep movsd
  mov di,dx  ;picslave
  mov cx,8
  rep movsd
  ret
irq_uninit endp

;this proc was copied almost directly from PMODE v3.07 - why reinvent the
;wheel?
int3h proc ;pmode
  ;this will redir INTs (and IRQs) to rmode
  cmp dptr[esp],offset ints            ;I added these extra checks in case
  jb do3                               ;this kernel is debugged later
  cmp dptr[esp],offset ints+255
  ja do3
  cmp wptr[esp+4],selcode16            ;was it from our ints table or is it
  jz redir                             ; an embedded int 3?
;continue on to normal INT 3 handler
do3:
  jmp fptr cs:int3vector               ; JMP FAR sel:off32 handler
;redirect to proper handler in rmode
redir:
  mov [esp+8],eax                      ; store EAX for later POPAD
  mov eax,[esp]                        ; get address in redirection matrix
  add esp,8                            ; discard EIP and CS from INT 3
  push ecx                             ;-+
  push edx                             ; |
  push ebx                             ; |
  push esp                             ; | (ignored by CPU)
  push ebp                             ; |
  push esi                             ; |
  push edi                             ;store rest of registers for POPAD
  push ds                              ;-+
  push es                              ; |
  push fs                              ; |
  push gs                              ;store all segment registers

  mov ds,cs:_selzero32                 ; DS -> 0 (beginning of memory)
  mov edi,cs:prg_base16                ; EDI = offset of SEGS16 from 0

  mov dx,cs:pm2rm_stkcur               ; DX = SS for real mode redirection
  xor ebp,ebp                          ;-+
  mov bp,dx                            ; |
  shl ebp,4                            ; EBP -> top of real mode stack

  cmp dx,cs:pm2rm_stkend               ; exceeded real mode stack space?
  jz critical_error                    ; if yes, critical error (hang)

  mov bx,cs:pm2rm_stklen               ; get size of real mode stack
  sub dx,bx                            ; adjust DX to next stack location
  sub ds:pm2rm_stkcur[edi],bx          ; update ptr for possible reenterancy
  shl bx,4                             ; set real mode SP to top of stack

  mov ds:[ebp-2],ss                    ; store SS:ESP on real mode stack
  mov ds:[ebp-6],esp

  sub ax,offset ints+1                 ; AX = int number
                                       ; the +1 is because the next instruct
                                       ; to return to is after the one-byte
                                       ; int 3 opcode
  mov ah,al                            ;-+
  and ah,0f8h                          ; AH = high 5 bits of int number

  cmp ah,cs:picmaster                  ; low IRQ?
  jz redir_irq                         ; if yes, do IRQ
  cmp ah,cs:picslave                   ; high IRQ?
  jnz redir_int                        ; if no, do INT (with general regs)
  add al,8                             ; upper 8 IRQs
;-----------------------------------------------------------------------------
redir_irq:                             ; an IRQ redirection
  mov si,segs16                        ; real mode target CS:IP
  mov di,offset r1       
  sub bx,6                             ; adjust real mode SP for stored vars
  sub al,ah                            ; convert INT # to IRQ #
  xor ah,ah
  shl ax,2
  mov bp,offset old_irqs
  add bp,ax
  mov ax,segs16                        ; rmode DS

  db 66h                               ; JMP DWORD PTR, as in 32bit offset,
  jmp wptr cs:pm_2_rm                  ;  not seg:16bit offset

r1:
  pushf
  push cs
  push offset r1b
  push dptr ds:[bp]
  retf
r1b:
  mov ax,seldata16                     ; DS selector value for protected mode
  mov cx,ax                            ; ES selector value for protected mode
  pop ebx                              ; get protected mode SS:ESP from stack
  pop dx
  mov si,selcode16                     ; target CS:EIP in protected mode
  mov edi,offset r2

  jmp cs:rm_2_pm                       ; go back to protected mode

r2:
  mov ax,pm2rm_stklen                  ; release real mode stack
  add pm2rm_stkcur,ax

  pop gs                               ;-+
  pop fs                               ; |
  pop es                               ; |
  pop ds                               ; |
  popad                                ;restore all registers
  iretd

;-----------------------------------------------------------------------------
redir_int:                             ; an INT redirection
  mov ds:intnum[edi],al                ; modify code with interrupt number

  mov es,cs:_selzero32                 ; copy registers from protected mode
  lea edi,[ebp-26h]                    ;  stack to real mode stack
  lea esi,[esp+8]                      ; esp+8 => popad
  mov ecx,8
  cld
  rep movsd es:[edi],ss:[esi]          ; note the SS override!

  mov si,segs16                        ; real mode target CS:IP
  mov di,offset i1
  sub bx,26h                           ; adjust real mode SP for stored vars

  db 66h                               ; JMP DWORD PTR, as in 32bit offset,
  jmp wptr cs:pm_2_rm                  ;  not seg:16bit offset

i1:
  popad                                ; load regs with int call values

  db 0cdh                              ; INT intnum
intnum db ?

  pushad                               ; store registers on stack
  pushf                                ; store flags on stack
  cli
  cld

  xor eax,eax                          ; EAX = linear ptr to SS
  mov ax,ss
  shl eax,4
  xor ebp,ebp                          ;-+
  mov bp,sp                            ; EBP = SP

  mov ebx,[bp+22h]                     ; get protected mode SS:ESP from stack
  mov dx,[bp+26h]

  add ebp,eax                          ; EBP -> stored regs on stack

  mov ax,selzero32                     ; DS selector value for pmode zero32
  mov cx,dx                            ; ES selector value for pmode stack
  mov si,selcode16                     ; target CS:EIP in protected mode
  mov edi,offset i2

  jmp cs:rm_2_pm                       ; go back to protected mode

i2:
  mov ax,ds:[ebp]                      ; move return FLAGS from real mode
  and ax,8d5h                          ;  stack to protected mode stack
  mov bx,[esp+30h]                     ; get flags from stack
  and bx,not 8d5h
  or ax,bx                             ; merge pmode/rmode flags
  mov [esp+30h],ax                     ; put flags onto IRETD

  lea esi,[ebp+2]                      ;DS:ESI => rmode pushad
  lea edi,[esp+8]                      ;ES:EDI => pmode popad
  mov ecx,8
  db 67h                               ;-+
  rep movsd                            ; copy from rmode stack to pmode stack

  mov edi,cs:prg_base16                ; release  real mode stack
  mov ax,cs:pm2rm_stklen
  add ds:pm2rm_stkcur[edi],ax

  pop gs                               ;-+
  pop fs                               ; |
  pop es                               ; |
  pop ds                               ; |
  popad                                ; restore all regs
  iretd
int3h endp

_000h proc  ;DPMI func 0h (alloc descriptors)
  ;CX=# descriptors wanted
  push ds
  pushad
  .if !cx
    jmp fail
  .endif
  mov ds,cs:_selzero32
  mov esi,cs:gdt_addr
  mov edi,cs:gdt_end
  xor bx,bx
  xor dx,dx  ;current counter
top:
  .if esi==edi
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  cmp bptr[esi+5],0
  jz found
  add esi,8
  inc bx
  jmp top
found:  ;found a free desc
  mov ebp,esi
loop1:
  add dx,8
  add esi,8
  .if dx==cx
    mov edi,ebp
loop2:
    mov bptr[edi+5],10010010b  ;data, expand-up, writeable, PL=0
    add edi,8
    dec cx
    jnz loop2
    mov wptr[esp].callstruct._eax,bx
    popad
    pop ds
    and bptr[esp+8],0feh  ;clear carry
    iretd
  .endif
  .if esi==edi
    jmp fail
  .endif
  cmp bptr[esi+5],0
  jz loop1
  add bx,dx
  xor dx,dx
  jmp top
_000h endp

_001h proc  ;free desc (bx=selector)
  push ds
  pushad
  mov ds,cs:_selzero32
  mov cx,cs:sels
  add cx,12
  cmp bx,cx
  .if carry? || (bx & 7)  ;MASM checks left to right
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd      
  .endif
  mov edi,cs:gdt_addr
  xor eax,eax
  mov ax,bx
  add edi,eax
  xor ax,ax
  stosd
  stosd
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_001h endp

_002h proc  ;segment to desc (bx=segment)
  push ds
  pushad
  mov ds,cs:_selzero32
;first see if one is already setup for this segment
  mov esi,cs:gdt_addr
  mov edi,cs:gdt_end
  xor dx,dx
top:
  .if esi==edi
    jmp makenew
  .endif
  cmp bptr[esi+6],10h  ;is AVL bit set?
  .if zero?
    mov ecx,[esi+2]
    shr ecx,4
    .if cx==bx
      mov wptr[esp].callstruct._eax,dx
      jmp ok
    .endif
  .endif
  add esi,8
  add dx,8
  jmp top
;nope:so make a new one
makenew:
  xor ax,ax
  mov cx,1
  xor edx,edx
  mov dx,bx  ;preserve for later
  int 31h  ;alloc a desc
  .if carry?
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  ;bx=selector
  mov wptr[esp].callstruct._eax,bx
  mov edi,cs:gdt_addr
  ;setup selector as needed
  xor ecx,ecx
  mov cx,bx
  add edi,ecx
  mov wptr[edi+0],0ffffh  ;limit = 64k
  shl edx,4
  or dptr[edi+2],edx      ;set base (must preserve 4th byte = type setup)
  or wptr[edi+6],10h      ;set the AVL bit so I can use it again if need be
ok:
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_002h endp

_003h proc  ;set inc value (always 8)
  mov ax,8
  add bptr[esp+8],0feh
  iretd
_003h endp

_006h proc ; get desc base (bx=sel) (cx:dx=base)
  push ds
  pushad
  mov ds,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  xor esi,esi
  mov si,bx
  add esi,cs:gdt_addr
  .if esi>=cs:gdt_end
    jmp fail
  .endif
  mov dx,[esi+2]   ;get base 15..0
  mov cl,[esi+3]   ;get base 23..16
  mov ch,[esi+7]   ;get base 31..24
  mov wptr[esp].callstruct._ecx,cx
  mov wptr[esp].callstruct._edx,dx
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_006h endp

_007h proc ; set desc base (bx=sel cx:dx=base)
  push ds
  pushad
  mov ds,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  xor esi,esi
  mov si,bx
  add esi,cs:gdt_addr
  .if esi>=cs:gdt_end
    jmp fail
  .endif
  mov [esi+2],dx   ;set base 15..0
  mov [esi+3],cl   ;set base 23..16
  mov [esi+7],ch   ;set base 31..24
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_007h endp

_008h proc  ;set desc limit (bx=sel cx:dx=limit)
  push ds
  pushad
  mov ds,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  xor esi,esi
  mov si,bx
  add esi,cs:gdt_addr
  .if esi>=cs:gdt_end
    jmp fail
  .endif
  shl ecx,16
  mov cx,dx
  .if ecx>(1024*1024-1)
    or bptr[esi+6],80h  ;set the G bit
    shr ecx,20
  .else
    and bptr[esi+6],7fh ;clear the G bit
  .endif
  mov [esi],cx
  shr ecx,8
  or [esi+6],ch
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_008h endp

_009h proc  ;change desc access right word (bx=sel cx=rights)
  push ds
  pushad
  mov ds,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  xor esi,esi
  mov si,bx
  add esi,cs:gdt_addr
  .if esi>=cs:gdt_end
    jmp fail
  .endif
  mov [esi+5],cx
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_009h endp

_00ah proc  ;create alias desc (bx=sel)
  push es
  pushad
  sub esp,8  ;reserve room for info
  push ss
  pop es
  mov edi,esp
  mov ax,0bh  ;get desc
  int 31h
  .if carry?
fail:
    add esp,8
    popad
    pop es
    or bptr[esp+8],1
    iretd
  .endif
  xor ax,ax
  mov cx,1
  int 31h  ;alloc a desc ;ax=desc
  jc fail 
  mov bx,ax
  mov ax,0ch
  int 31h     ;set desc
  jc fail
  add esp,8
  mov wptr [esp].callstruct._eax,bx
  popad
  pop es
  and bptr[esp+8],0feh
  iretd
_00ah endp

_00bh proc  ;get desc (bx=sel  ES:EDI=buffer)
  push ds
  pushad
  mov ds,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  xor esi,esi
  mov si,bx
  add esi,cs:gdt_addr
  .if esi>=cs:gdt_end
    jmp fail
  .endif
  mov cx,8
  db 67h
  rep movsb
  popad
  pop ds
  and bptr[esp+8],0feh
  iretd
_00bh endp

_00ch proc  ;set desc (bx=sel  ES:EDI=buffer)
  push ds
  push es
  pushad
  push es
  pop ds
  mov es,cs:_seldata32
  .if bx & 7
fail:
    popad
    pop es
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  mov esi,edi
  xor edi,edi
  mov di,bx
  add edi,cs:gdt_addr
  .if edi>=cs:gdt_end
    jmp fail
  .endif
  mov cx,8
  db 67h
  rep movsb
  popad
  pop es
  pop ds
  and bptr[esp+8],0feh
  iretd
_00ch endp

_100h proc  ;alloc DOS ram (func 48h) and a selector for it (bx=paras)
;BUG : this func needs lots of help
  pushad
  sub esp,sizeof callstruct
  mov dptr [esp].callstruct._sp,0  ;clear SS:SP
  mov bptr [esp+1].callstruct._eax,48h
  mov ax,300h
  mov bx,21h
  xor cx,cx
  int 31h
  .if carry?
fail:
    add esp,sizeof callstruct
fail2:
    popad
    or bptr[esp+8],1
    iretd
fail3:  ;DOS error code returned
    mov ax,wptr[esp].callstruct._eax
    mov bx,wptr[esp].callstruct._ebx
    add esp,sizeof callstruct
    mov wptr[esp].callstruct._eax,ax
    mov wptr[esp].callstruct._ebx,bx
    jmp fail2
  .endif
  test bptr[esp].callstruct._flg,1  ;carry set?
  jnz fail3
  ;bx=segment
  mov bx,wptr[esp].callstruct._ebx
  add esp,sizeof callstruct
  mov ax,2
  int 31h
  jc fail2
  ;ax=selector
  mov wptr [esp].callstruct._eax,bx  ;save AX (segment)
  mov wptr [esp].callstruct._edx,ax  ;save DX (selector)
  popad
  and bptr[esp+8],0feh
  iretd
_100h endp

_101h proc  ;free DOS mem (DX=sel)
  pushad
  mov bx,dx
  mov ax,6
  int 31h
  .if carry?
fail:
    popad
    or bptr[esp+8],1
    iretd
fail2:
    add esp,sizeof callstruct
    jmp fail
fail3:  ;DOS error code (ax)
    mov ax,wptr[esp].callstruct._eax
    add esp,sizeof callstruct
    mov wptr[esp].callstruct._eax,ax
    jmp fail
  .endif
  shl ecx,16
  mov cx,dx
  ;ECX = segment
  shr ecx,4  ;convert to segment
  mov ax,1  ;free desc
  int 31h
  jc fail
  sub esp,sizeof callstruct
  mov dptr[esp].callstruct._sp,0  ;clear SS:SP
  mov [esp].callstruct._es,cx
  mov ax,300h
  mov bx,21h
  xor cx,cx
  mov edi,esp
  push es
  push ss
  pop es
  int 31h
  pop es
  jc fail2
  test bptr[esp].callstruct._flg,1
  jnz fail3
  add esp,sizeof callstruct
  popad
  and bptr[esp+8],0feh
  iretd
_101h endp

_102h proc  ;resize DOS mem (bx=new size  dx=sel)
  pushad
  sub esp,sizeof callstruct
  mov dptr[esp].callstruct._sp,0  ;clear SS:SP
  mov bptr[esp+1].callstruct._eax,4ah  ;resize mem block
  mov ax,300h
  mov bx,21h
  xor cx,cx
  push es
  push ss
  pop es
  mov edi,esp
  int 31h
  pop es
  .if carry?
fail:
    add esp,sizeof callstruct
fail3:
    popad
    or bptr[esp+8],1
    iretd
fail2:
    mov ax,wptr[esp].callstruct._eax
    add esp,sizeof callstruct
    mov wptr[esp].callstruct._eax,ax
    jmp fail3
  .endif
  test bptr[esp].callstruct._flg,1
  jnz fail2
  add esp,sizeof callstruct
  popad
  and bptr[esp+8],0feh
  iretd
_102h endp

_200h proc  ;get Rmode vektor (bl=#) (cx:dx)
  push ds
  mov ds,cs:_selzero32
  push eax
  xor eax,eax
  mov al,bl
  shl eax,2  ;*4
  mov cx,[eax+2]  ;get segment
  mov dx,[eax]    ;get offset
  pop eax
  pop ds
  and bptr[esp+8],0feh
  iretd
_200h endp

_201h proc  ;set Rmode vektor (bl=# cx:dx)
  push ds
  mov ds,cs:_selzero32
  push eax
  xor eax,eax
  mov al,bl
  shl eax,2  ;*4
cli
  mov [eax+2],cx  ;set segment
  mov [eax],dx    ;set offset
sti
  pop eax
  pop ds
  and bptr[esp+8],0feh
  iretd
_201h endp

_202h proc  ;get exc handler (bl=# 0-1fh) (cx:edx=handler)
  .if bl > 01fh
    or bptr[esp+8],1
    iretd
  .endif
  jmp _204h  ;NOTE : For now this is the same as 204h but it will change
             ;when I add exc handlers
_202h endp

_203h proc  ;set exc handler (bl=# cx:edx=handler)
  .if bl > 01fh
    or bptr[esp+8],1
    iretd
  .endif
  jmp _205h  ;NOTE : For now this is the same as 205h but it will change
             ;when I add exc handlers
_203h endp

_204h proc
  ;in : bl= INT #
  ;out: CX:EDX = vektor
  push ds
  pushad
  mov ds,cs:_seldata16
  xor eax,eax
  mov al,bl
  shl eax,3  ;*8
  mov esi,offset idt
  add esi,eax
  mov dx,[esi+6]                       ; get the offset (16-31)
  shl edx,16
  mov dx,[esi]                         ; get the offset (0-15)
  mov cx,[esi+2]                       ; get the selector
  mov wptr [esp].callstruct._ecx,cx
  mov dptr [esp].callstruct._edx,edx
  popad
  pop ds
  and bptr [esp+8],0feh  ;clear carry
  iretd
_204h endp

_205h proc
  ;in : bl= INT # CX:EDX = vektor
  push ds
  pushad
  mov ds,cs:_seldata16
  xor eax,eax
  mov al,bl
  shl eax,3  ;*8
  mov esi,offset idt
  add esi,eax
  cli
  mov [esi],dx                   ; set the offset (0-15)
  shr edx,16
  mov [esi+6],dx                 ; set the offset (16-31)
  mov [esi+2],cx                 ; set the selector

;set IRQSET if needed
  mov dl,bl
  and dl,0f8h  ;mask high 5 bits
  cmp dl,picmaster
  jnz i1
  jmp s1
i1:
  cmp dl,picslave
  jnz i2
  add bl,8
s1:
  sub bl,dl
  mov cl,bl
  mov dx,1
  shl dx,cl
  or irqset,dx
i2:

  popad
  pop ds
  and bptr [esp+8],0feh  ;clear carry
  iretd
_205h endp

;-----------------------------------------------------------------------------
;int 31h funcs 300h,301h,302h
_300h proc
  ;300h = simulate rmode INT
  ;301h = call CS:IP with RETF stack frame
  ;302h = call CS:IP with IRET stack frame
  ; in : ES:(E)DI => callstruct
  ;    : BH = 0
  ;    : BL = int # (if 300h)
  ;    : CX = # words to copy to stack
  ; out: ES:(E)DI => updated callstruct

;if you are going to trace this code I suggest you grab paper and put the
;rmode and pmode stacks on it and create them as you look thru this

  std                                  ;copy backwards
  pushad
  mov ebp,esp
  add ebp,11*4                         ;ebp -> stack params (pushad:EIP:CS:flags)
  push ds
  push es
  push fs
  push gs

  mov ds,cs:_seldata16
  mov ebx,edi
  cmp dptr es:[edi].callstruct._sp,0
  jz s1
;get SS:SP from callstruct
  xor eax,eax
  mov ax,es:[edi].callstruct._ss
  push ax                              ;save for later (rmode SS)
  shl eax,4
  xor edx,edx
  mov dx,es:[edi].callstruct._sp
  push dx                              ;save for later (rmode SP)
  add eax,edx
  jmp c1                               ; EAX = rmode TOS, DX = SP
s1:  ;supply system stack
  xor eax,eax
  mov ax,pm2rm_stkcur                  ;get next stack location
  cmp ax,pm2rm_stkend                  ;check if a stack is avail.
  jnz ok1
  jmp int31hfail
ok1:
  mov dx,pm2rm_stklen                  ;make stack pt to next location
  sub ax,dx
  sub pm2rm_stkcur,dx
  push ax                              ;save for later (rmode SS)
  add ax,dx
  shl eax,4
  shl dx,4                             ; EAX = rmode TOS, DX = SP
  push dx                              ;save for later (rmode SP)
c1:
;preserve current rmode stack ptr (SP) (so it will be easy to pop off
; parms from rmode stack later)
  sub dx,6                             ;reserve space for SS:ESP on rmode stk
  xchg dx,old_sp
  push dx                              ;preserve old_sp
;prepare rmode stack ptr
  push es                              ;preserve ES
  mov es,_selzero32                    ;-+
  mov edi,eax                          ;es:edi => dest stack
;copy SS:ESP onto rmode stack so they can be restored later on
  sub edi,2
  mov es:[edi],ss
  sub edi,4
  lea eax,[esp+2]
  db 67h
  stosd
  add edi,2
;check if there are any stack params
  or cx,cx
  jz c2
;prepare pmode stack ptr
  push ss                              ;-+
  pop ds                               ; |
  mov esi,ebp                          ;ds:esi => src stack
;copy over CX words from current stack to new stack (params)
  xor edx,edx
  mov dx,cx
  mov ecx,edx                          ;clear high word of ecx
  dec dx
  shl edx,1
  add esi,edx
  db 67h                               ;-+
  rep movsw                            ;copy stack over
  add esi,2                            ;move back to normal pos
;copy (maybe Flags):CS:IP to rmode stack (used to return back to pmode)
c2:
  cmp bptr[ebp-11*4].callstruct._eax,1
  jz f4
  pushf
  pop ax
  db 67h
  stosw
f4:
  mov ax,segs16
  db 67h
  stosw 
  mov ax,offset r2
  db 67h
  stosw
;copy Flags:CS:IP to call rmode int/proc
  pop ds                               ;restore ES into DS
  mov ax,[ebx].callstruct._flg         ;flags from callstruct
  cmp bptr[ebp-11*4].callstruct._eax,1
  jz f1
  and ah,0fch                          ;clear IF and TF for func 300h,302h
f1:
  db 67h
  stosw                                ;save flags on rmode stack
  cmp bptr[ebp-11*4].callstruct._eax,0
  jnz f2
;300h
  xor eax,eax
  mov al,bptr[ebp-11*4].callstruct._ebx
  shl eax,2  ;*4
  mov eax,es:[eax]                     ;get rmode INT # CS:IP
  jmp f3
f2:
;301h and 302h
  mov eax,dptr[ebx].callstruct._ip
f3:
  sub edi,2
  db 67h
  stosd
  add edi,2
;copy FS and GS for real mode
  mov ax,[ebx].callstruct._gs
  db 67h
  stosw
  mov ax,[ebx].callstruct._fs
  db 67h
  stosw
;copy over callstruct (only what is needed)
  sub edi,2
  mov ecx,8                            ;8 dwords to copy
  mov esi,ebx                          ;ds:esi => callstruct
  add esi,7*4                          ;start from bottom of block
  db 67h                               ;-+
  rep movsd                            ;copy callstruct over
  add esi,4                            ;ds:esi => callstruct
  add edi,4                            ;es:edi => rmode tos

;  AX = new DS
;  CX = new ES
;  DX = new SS
;  BX = new SP
;  SI = new CS
;  DI = new IP

  xor eax,eax
  mov dx,[esp+4]                       ;SS
  mov ax,dx
  shl eax,4
  sub edi,eax
  mov bx,di                            ;SP

  mov ax,[esi].callstruct._ds          ;DS
  mov cx,[esi].callstruct._es          ;ES
  mov si,segs16                        ;CS
  mov di,offset r1                     ;IP

  db 66h                               ;-+
  jmp wptr cs:pm_2_rm                  ;jmp DWORD PTR  (32bit offset)

r1:

  popad
  pop fs
  pop gs

  iret

r2:    ;return back to PMODE
;remove stack parameters
  mov sp,cs:old_sp
  push gs                              ;order preserved is important
  push fs
  push ds
  push es
  pushf
  pushad

  mov bp,sp
  mov ebx,[bp+42]                      ;for ESP
  mov dx,[bp+46]                       ;for SS
  mov si,selcode16
  mov ax,selzero32
  mov cx,ax
  mov edi,offset p1

;  AX  = new DS
;  CX  = new ES
;  DX  = new SS
;  EBX = new ESP
;  SI  = new CS
;  EDI = new EIP

  jmp cs:rm_2_pm

p1:
  mov ebx,cs:prg_base16
  pop ds:old_sp[ebx]

  mov edi,[esp+6*2].callstruct._edi  ;get saved EDI
  mov es,[esp+4*2]                   ;get saved ES

;copy rmode stack to callstruct
  xor eax,eax
  pop ax                               ;pop rmode SP
  xor esi,esi
  pop si                               ;pop rmode SS
  shl esi,4
  add esi,eax
  sub esi,42+6                         ;move back to saved regs on rmode stack
  mov ecx,10
  cld
  mov eax,es:[edi].callstruct._res     ;this must be reserved
  mov [esi].callstruct._res,eax
  db 67h
  rep movsd
  db 67h                               ;copy last word
  movsw
;release system stack
  cmp dptr es:[edi].callstruct._sp,0
  jz p4
  mov ax,ds:pm2rm_stklen[ebx]
  add ds:pm2rm_stkcur[ebx],ax          ;stack released
p4:
  pop gs
  pop fs
  pop es
  pop ds
  popad
  and bptr[esp+8],0feh                 ;clear carry flag = sucessful
  iretd
_300h endp

;-----------------------------------------------------------------------------
; 303h DPMI func (alloc callback)
_303h proc
  pushad
  push ds
  push es
  push fs
  push gs
  mov bl,cs:callbacks
  or bl,bl
  jz int31hfail

  mov ds,cs:_selzero32
  mov edx,cs:callbackbase
  mov ecx,edx
chk1:
  cmp wptr[edx+3],0
  jz free
  add edx,25
  dec bl
  jnz chk1
  jmp int31hfail
free:
  mov bx,[esp+6]                       ; BX = caller DS from stack
  mov [edx+3],bx                       ; store callback parms in callback
  mov [edx+7],esi
  mov [edx+12],es
  mov [edx+16],edi

  sub edx,ecx                          ; DX = offset of callback
  shr ecx,4                            ; CX = segment of callback

  mov wptr[esp+4*2].callstruct._edx,dx
  mov wptr[esp+4*2].callstruct._ecx,cx
  mov wptr[esp+4*2].callstruct._eax,ax

  pop gs
  pop fs
  pop es
  pop ds
  popad
  and bptr[esp+8],0feh                 ;clear carry flag = sucessful
  iretd

_303h endp

;-----------------------------------------------------------------------------
; 304h DPMI func
_304h proc
  pushad
  push ds
  push es
  push fs
  push gs
  cmp cx,cs:callbackseg
  jne int31hfail

  xor ebx,ebx
  mov bx,dx                            ; EBX = offset of callback

  xor ax,ax                            ; check if valid offset
  xchg dx,ax
  mov cx,25
  div cx

  or dx,dx                             ; is there a remainder
  jnz int31hfail                       ; if yes, not valid

  or ah,ah                             ; callback index too big?
  jnz int31hfail                       ; if yes, not valid

  cmp al,cs:callbacks                  ; callback index out of range?
  jae int31hfail                       ; if yes, not valid

  add ebx,cs:callbackbase              ; EBX -> callback
  mov word ptr [ebx+3],0               ; set callback as free

  pop gs
  pop fs
  pop es
  pop ds
  popad
  and bptr[esp+8],0feh                 ;clear carry flag = sucessful
  iretd

_304h endp

int31hfail:     ;Used in 30xh funcs (7)
  pop gs
  pop fs
  pop es
  pop ds
  popad
  or bptr[esp+8],1h                    ;set flag = failed
  iretd

;-----------------------------------------------------------------------------
; int 31h func 305h  (raw/xms)

_305h proc  ;get raw mode state save/restore addr's
;   AX     = size of buffer in bytes required to save state
;   BX:CX  = segment:offset of real mode routine used to save/restore state
;   SI:EDI = selector:offset of protected mode routine used to save/restore
;            state
  xor ax,ax
  mov bx,segs16
  mov cx,cs:sr_state_rm
  mov si,selcode16
  mov edi,cs:sr_state_pm
  and bptr[esp+11],0feh  ;clear carry
  iretd
_305h endp

_306h proc   ;get raw mode switching addr's
;   BX:CX  = segment:offset of real to protected mode switch procedure
;   SI:EDI = selector:offset of protected to real mode switch procedure
  mov bx,segs16
  mov cx,cs:rm_2_pm
  mov si,selcode16
  mov edi,cs:pm_2_rm
  and bptr[esp+11],0feh  ;clear carry
  iretd
_306h endp

_400h proc ;get version (see DPMI.DOC)
  mov ah,0
  mov al,5ah  ;v0.90!
  mov bx,1   ;host is 32bit
  .if cs:mode==MODE_VCPI
    or bx,2  ;v86 mode used
  .endif
  mov cl,cs:cpu
  mov dh,cs:picmaster
  mov dl,cs:picslave
  and bptr[esp+8],0feh
  iretd
_400h endp

include _ram.asm  ;funcs 500h-503h

_800h proc  ;bx:cx = phys addr  si:di=size
  .if cs:mode!=MODE_VCPI
success:
    and bptr[esp+8],0feh
    iretd
  .endif
  push ds
  pushad
  mov ds,cs:_selzero32
  shl ebx,16
  mov bx,cx
  shl esi,16
  mov si,di
  mov ecx,esi
; EBX=phys ECX=size
  mov esi,cs:pagetablefree
  mov edi,cs:pagetabletop  ;HERE!
t1:
  .if esi==edi
fail:
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
  cmp dptr[esi],0
  jz found
  add esi,4
  jmp t1
found:
  mov ebp,esi
  mov edx,4096  ;running total
t2:
  .if edx>=ecx
    ;setup page entries to point to phys addr (EBX)
    mov edi,ebp  ;move EDI till it reaches ESI
    mov edx,ebx
    and ebx,0fffff000h  ;keep bits 31-12 (11-0 = cleared)
    mov bl,7            ;set as user/writable/etc.
t3:
    mov [edi],ebx
    add ebx,4096
    add edi,4
    cmp edi,esi
    jnz t3
    sub edi,4  ;move back to last entry
    or bptr[edi+1],20h  ;set AVL bit 0 (indicates end of Mapping - used in 801h)

    and edx,0fffh       ;keep bits 11-0
    sub ebp,cs:pagetablebase
    add ebp,edx         ;add offset to linear addr
    ;shr ebp,2  ;/4
    ;shl ebp,12 ;*4k
    shl ebp,10
    add ebp,edx         ;add offset
    mov wptr[esp].callstruct._ecx,bp
    shl ebp,16
    mov wptr[esp].callstruct._ebx,bp
    popad
    pop ds
    and bptr[esp+8],0feh
    iretd
  .endif
  add esi,4
  .if esi==edi
    jmp fail
  .endif
  add edx,4096
  cmp dptr[esi],0
  jz t2
  jmp t1
  popad
  pop ds
_800h endp

_801h proc  ;free phys mapping (BX:CX = linear addr)
  .if cs:mode!=MODE_VCPI
    and bptr[esp+8],0feh
    iretd
  .endif

  push ds
  pushad
  mov ds,cs:_selzero32
  shl ebx,16
  mov bx,cx
  and ebx,0fffff000h  ;mask off 12 bits
  add ebx,cs:pagetablebase
  .if ebx < cs:pagetablefree || ebx > cs:pagetabletop
    popad
    pop ds
    or bptr[esp+8],1
    iretd
  .endif
t1:
  test bptr[ebx+1],10h  ;check AVL bit 0
  jnz lastone
  mov dptr[ebx],0
  add ebx,4
  jmp t1
lastone:
  mov dptr[ebx],0
  popad
  pop ds
  and bptr[esp+8],0feh
  iret
_801h endp

_900h proc  ;get & dis IF
  push eax
  pushf
  pop ax
  cli
  shr ah,1
  and ah,1
  mov bptr[esp],ah
  pop eax
  and bptr[esp+8],0feh
  iretd
_900h endp

_901h proc  ;get & en IF
  push eax
  pushf
  pop ax
  sti
  shr ah,1
  and ah,1
  mov bptr[esp],ah
  pop eax
  and bptr[esp+8],0feh
  iretd
_901h endp

_902h proc  ;get IF
  push eax
  pushf
  pop ax
  shr ah,1
  and ah,1
  mov bptr[esp],ah
  pop eax
  and bptr[esp+8],0feh
  iretd
_902h endp

_eeffh proc  ;DOS Extender Call
  mov eax,514d3332h   ;'QM32' But you can't use cmp eax,'QM32'
                      ;       You would use cmp eax,'23MQ' because MASM
                      ;        will reverse the order of the letters.
  mov es,cs:_selzero32
  mov ebx,offset msg_copyright
  add ebx,cs:prg_base16
  mov ch,cs:mode
  mov cl,cs:cpu
  mov dh,VERSION_HI
  mov dl,VERSION_LO
  and bptr[esp+8],0feh
  iretd
_eeffh endp

int31h_table_start equ $
int31h_table label word
  dw 0,1,2,3,6,7,8,9,0ah,0bh,0ch
  dw 100h,101h,102h
  dw 200h,201h,202h,203h,204h,205h
  dw 300h,301h,302h,303h,304h,305h,306h
  dw 400h
  dw 500h,501h,502h,503h
  dw 800h,801h
  dw 900h,901h,902h
  dw 0eeffh
int31h_table_size equ ($-int31h_table_start)/2
int31h_offset label word
  dw offset _000h
  dw offset _001h
  dw offset _002h
  dw offset _003h
  dw offset _006h
  dw offset _007h
  dw offset _008h
  dw offset _009h
  dw offset _00ah
  dw offset _00bh
  dw offset _00ch
;-----------------
  dw offset _100h
  dw offset _101h
  dw offset _102h
;-----------------
  dw offset _200h
  dw offset _201h
  dw offset _202h
  dw offset _203h
  dw offset _204h
  dw offset _205h
;-----------------
  dw offset _300h    ;NOTE : Func 300h,301h and 302h all go to one func
  dw offset _300h    ; "
  dw offset _300h    ; "
  dw offset _303h
  dw offset _304h
  dw offset _305h
  dw offset _306h
;-----------------
  dw offset _400h
;-----------------
  dw offset _500h
  dw offset _501h
  dw offset _502h
  dw offset _503h
;-----------------
  dw offset _800h
  dw offset _801h
;-----------------
  dw offset _900h
  dw offset _901h
  dw offset _902h
;-----------------
  dw offset _eeffh
;-----------------
int31h proc
  ;ax=func #
  push edi
  push ecx
  push es
  cld
  mov es,cs:_seldata16
  mov di,offset int31h_table
  mov ecx,int31h_table_size
  repnz scasw
  jnz bad
  mov edi,int31h_table_size
  sub edi,ecx
  dec edi
  shl edi,1
  add edi,offset int31h_offset
  mov di,es:[edi]
  pop es
  pop ecx
  shl edi,16         ;move into high word so we can skip over low word
  xchg edi,[esp]     ;pop edi, store RET addr
  add esp,2          ;skip over low word (it's only a 16bit addr)
  ret                ;jmp to function handler
bad:
  pop es
  pop ecx
  pop edi
  ; EIP , CS , EFlags
  or bptr[esp+8],1  ;set carry
  iretd
int31h endp

;-----------------------------------------------------------------------------
; Callback handler
callback proc                          ; real mode callback handler
  mov ax,sp                            ; preserve SS:SP for callback
  push ss
  push ax

  push gs                              ; preserve real mode regs for callback
  push fs
  push ds
  push es
  pushf                                ; preserve FLAGS for callback

  cli
  cld

  mov ebp,cs:rm2pm_stkcur              ; EBP = ESP for protected mode

  mov ebx,ebp                          ; set EBX to next stack location
  sub ebx,cs:rm2pm_stklen 
  mov cs:rm2pm_stkcur,ebx              ; update ptr for possible reenterancy

  cmp ebx,cs:rm2pm_stkend              ; exceeded protected mode stack space?
  jz critical_error                    ; if yes, critical error (hang)

  xor eax,eax                          ; EAX = base address of SS
  mov ax,ss
  shl eax,4

  xor ebx,ebx                          ;-+
  mov bx,sp                            ; EBX = current linear SS:SP
  add ebx,eax

  or eax,92000000h                     ;-+ 
  mov edi,gdt_start+selcallbackds+2    ; set for protected mode callback DS
  mov cs:[edi],eax                     ; base address in GDT

  mov ax,selzero32                     ; DS selector for protected mode
  mov dx,ax                            ; SS selector = DS selector
  mov si,selcode16                     ; target protected mode CS:EIP
  mov edi,offset cb0

  jmp cs:rm_2_pm                       ; go to protected mode

cb0:
  mov edi,[esp+14]                     ; EDI -> register structure from stack

  lea esi,[esp+24]                     ; copy general registers from stack
  mov ecx,8                            ;  to register structure
  db 67h
  rep movsd

  mov esi,esp                          ; copy FLAGS, ES, DS, FG, and GS
  db 67h
  movsw
  db 67h
  movsd
  db 67h
  movsd

  db 67h
  lodsd                                ; EAX = real mode SS:SP from stack
  add ax,42                            ; adjust SP for stuff on stack
  mov es:[edi+4],eax                   ; put in register structure

  mov ds,cs:_selcallbackds             ; DS = callback DS selector
  sub edi,42                           ; EDI -> register structure
  xor esi,esi                          ;-+
  mov si,ax                            ; ESI = old real mode SP
  xchg esp,ebp                         ; ESP = protected mode stack

  pushfd                               ; push flags for IRET from callback
  db 66h                               ;-+
  push cs                              ; push 32bit CS for IRET
  db 66h                               ;-+
  push offset cb1                      ; |
  dw 0                                 ; push 32bit EIP for IRET

  xor eax,eax
  mov ax,word ptr [ebp+22]             ; EAX = target CS of callback
  push eax                             ; push 32bit CS for RETF to callback
  push dword ptr [ebp+18]              ; push 32bit EIP for retf

  db 66h                               ; 32bit RETF to callback
  retf

cb1:
  cli
  cld

  push es                              ; DS:ESI = register structure
  pop ds
  mov esi,edi

  mov es,cs:_selzero32                 ; ES -> 0 (beginning of memory)

  xor ebx,ebx
  mov bx,word ptr [esi+2eh]            ; EBX = real mode SP from structure
  xor edx,edx
  mov dx,word ptr [esi+30h]            ; EDX = real mode SS from structure
  sub bx,42                            ; subtract size of vars to be put

  mov ebp,[esi+0ch]                    ; EBP = pushed ESP from real mode
  mov bp,bx                            ; EBP = old high & new low word of ESP

  lea edi,[edx*4]                      ; EDI -> real mode base of stack
  lea edi,[edi*4+ebx]                  ;  of vars to be stored

  mov ecx,8                            ; copy general registers to stack
  db 67h
  rep movsd

  mov eax,[esi+6]                      ; EAX = return FS and GS for real mode
  mov es:[edi],eax                     ; store on real mode stack for return

  mov ax,[esi]                         ; AX = return FLAGS for real mode
  mov es:[edi+8],ax                    ; store on real mode stack for return
  mov eax,[esi+10]                     ; EAX = return CS:IP for real mode
  mov es:[edi+4],eax                   ; store on real mode stack for return

  mov ax,[esi+4]                       ; AX = return DS for real mode
  mov cx,[esi+2]                       ; CX = return ES for real mode

  mov si,segs16                        ; real mode target CS:IP
  mov di,offset cb2

  db 66h                               ; JMP DWORD PTR, as in 32bit offset,
  jmp wptr cs:pm_2_rm                  ;  not seg:16bit offset

cb2:
  mov esp,ebp                          ; restore total ESP, old high word

  mov eax,cs:rm2pm_stklen              ; restore top of protected mode stack
  add cs:rm2pm_stkcur,eax

  popad                                ; get callback return general regs
  pop fs                               ;-+
  pop gs                               ; get callback return FS and GS values
  iret                                 ; go to callback return CS:IP
callback endp

;-----------------------------------------------------------------------------
; Rmode IRQ redirectors

irq0:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0001h
  jmp rm_irq_redir

irq1:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0102h
  jmp rm_irq_redir

irq2:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0203h
  jmp rm_irq_redir

irq3:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0304h
  jmp rm_irq_redir

irq4:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0405h
  jmp rm_irq_redir

irq5:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0506h
  jmp rm_irq_redir

irq6:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0607h
  jmp rm_irq_redir

irq7:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0708h
  jmp rm_irq_redir

irq8:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0809h
  jmp rm_irq_redir

irq9:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,090ah
  jmp rm_irq_redir

irqa:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0a0bh
  jmp rm_irq_redir

irqb:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0b0ch
  jmp rm_irq_redir

irqc:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0c0dh
  jmp rm_irq_redir

irqd:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0d0eh
  jmp rm_irq_redir

irqe:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0e0fh
  jmp rm_irq_redir

irqf:
  sub esp,4   ;reserve room for RETF
  push cx
  mov cx,0f10h
  jmp rm_irq_redir

rm_irq_redir proc
  push eax
  mov ax,cs:irqset
r1:
  rcr ax,1
  dec cl
  jnz r1
  jc doit

;branch onto normal IRQ handler
  push bx
  mov bx,offset old_irqs
  xor ax,ax
  mov al,ch
  shl ax,2
  add bx,ax
  mov eax,cs:[bx]
  mov [esp+8],eax
  pop bx
  pop eax
  pop cx
  retf

;redir to PMODE
doit:
  mov cs:rm_irqnum,ch
  cmp ch,8
  jb low_irq
;high_irq
  mov al,cs:picslave
  sub al,8
  jmp r2
low_irq:
  mov al,cs:picmaster
r2:
  add cs:rm_irqnum,al
  pop eax
  pop cx
  add esp,4                            ;unreserve space for RETF
  pushad                               ;-+
  push ds                              ; |
  push es                              ; |
  push fs                              ; |
  push gs                              ; Preserve all regs

  mov ax,segs16
  mov ds,ax

  xor ebp,ebp     ;-+
  mov bp,ss       ; |
  shl ebp,4       ; |
  add ebp,esp     ;ebp => TOS
  mov ax,sp
  push ss         ;-+
  push ax         ;Save current stack pos

  mov ax,selzero32
  mov cx,ax
;alloc a RM 2 PM stack
  mov ebx,rm2pm_stkcur                 ; EBX = ESP (from 0)
  mov edx,rm2pm_stklen
  cmp ebx,rm2pm_stkend
  jz critical_error
  sub rm2pm_stkcur,edx
  mov dx,ax   ;selzero32
  mov si,selcode16
  mov edi,offset doit2

  jmp cs:rm_2_pm

doit2:   ;now in PMODE
  db 0cdh
rm_irqnum db ?

  mov dx,ds:[ebp-2]
  mov bx,ds:[ebp-4]
  mov si,segs16
  mov di,offset doit3

  db 66h
  jmp wptr cs:pm_2_rm

doit3:
;release stack used
  mov eax,cs:rm2pm_stklen
  add cs:rm2pm_stkcur,eax
;pop all values off and end IRQ
  pop gs
  pop fs
  pop es
  pop ds
  popad
  iret
rm_irq_redir endp

int21h_rm proc  ;INT 21h handler (rmode)
  cmp ah,4ch                           ; is program terminating?
  jz exit
normal:
  jmp cs:old_int21h_rm                 ;no, continue on to old int 21h handler
exit:                                  ;yes, program is terminating
  cmp wptr[esp+2],segs16               ;-+
  jnz normal                           ;make sure it's from our program
  push ax                              ;preserve error code and 4ch

  mov ax,segs16
  mov ds,ax
  xor ax,ax
  mov es,ax

  call irq_uninit

  mov eax,old_int21h_rm
  mov es:[21h*4],eax                   ;restore old INT 21h handler
  .if mode==MODE_VCPI
    call vcpi_uninit
  .elseif mode==MODE_XMS
    call xms_uninit
  .else
    call raw_uninit
  .endif
  pop ax
  jmp old_int21h_rm                    ; cont on to regular INT 21h (exit)
int21h_rm endp

code16seg ends

